#include "async_impl.h"
#include "error_code.h"
#include "utils.h"

#include <mail/nwsmtp/src/dkim/dkim_module.h>
#include <mail/nwsmtp/src/dkim/adkim.h>
#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/options.h>
#include <mail/nwsmtp/src/ratesrv/checks.h>

#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>

namespace NNwSmtp::NAsyncDlv {

namespace p = std::placeholders;

#include <boost/asio/yield.hpp>

void TAsyncDeliveryImpl::Delivery(TErrorCode ec, std::string answer) {
    auto& envelope = DlvRequest.Envelope;

    reenter(DeliveryCoroutine) {
        if  (!envelope->m_no_local_relay) {
            CurrentRcpts = GetRcpts([](DomainType domainType) {
                return !NUtil::IsRouting() || domainType == DomainType::LOCAL;
            });
            if (!CurrentRcpts.empty()) {
                yield Local();

                if (CanSendDsn(envelope, dsn::Options::SUCCESS)) {
                    yield Dsn(dsn::composer::success);
                }
            }
        }

        CurrentRcpts = GetRcpts([](DomainType domainType) {
            return !NUtil::IsRouting() || domainType == DomainType::LOCAL;
        });
        if (!CurrentRcpts.empty()) {
            yield Back();
        }

        if (NUtil::IsRouting()) {
            CurrentRcpts = GetRcpts([](DomainType domainType) {
                return domainType != DomainType::LOCAL;
            });
            if (!CurrentRcpts.empty()) {
                RemoveHeaders(envelope);

                if (!SenderDependentRelay.host.empty()) {
                    yield External(
                        SenderDependentRelay,
                        gconfig->delivery.senderDependent.timeouts
                    );
                } else if (!ForeignMxRelay.host.empty()) {
                    yield External(
                        ForeignMxRelay,
                        gconfig->delivery.external.timeouts
                    );
                } else if (!gconfig->delivery.external.addr.host.empty()) {
                    yield External(
                        gconfig->delivery.external.addr,
                        gconfig->delivery.external.timeouts
                    );
                } else {
                    NWLOG_L_EC(error, "EXTERNAL-SMTP", "Relay not set", EError::RelayNotSet);
                    return boost::asio::post(Io,
                        std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), EError::RelayNotSet, "554 5.4.4 Relay not set"));
                    
                }
            }

        }

        yield NRateSrv::AsyncUpdateCounters(
            Context,
            envelope->orig_message_size_,
            DlvRequest.Ip,
            envelope->m_sender_uid.empty() ? envelope->m_sender : envelope->m_sender_uid,
            envelope->m_sender_default_email.empty() ? envelope->m_sender : envelope->m_sender_default_email,
            envelope->m_rcpts,
            [this, self = shared_from_this()](auto) {
                boost::asio::post(Io, std::bind(&TAsyncDeliveryImpl::Delivery, self, TErrorCode{}, ""));
            }
        );

        tie(ec, answer) = CalculateResponse(envelope, CanSendDsn(envelope, dsn::Options::FAILURE));

        if (CanSendDsn(envelope, dsn::Options::FAILURE) && !ec) {
            DsnCoroutine = boost::asio::coroutine{};
            yield Dsn(dsn::composer::failure);
        }
        boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
    }
}

std::list<TRcpt> TAsyncDeliveryImpl::GetRcpts(std::function<bool(DomainType domainType)> pred) {
    auto rcpts =  boost::make_iterator_range(DlvRequest.Envelope->m_rcpts.begin(), DlvRequest.Envelope->m_rcpts.end())
        | boost::adaptors::filtered([&](const auto& rcpt) {
            return rcpt.m_delivery_status != check::CHK_ACCEPT
                && rcpt.m_delivery_status != check::CHK_REJECT
                && rcpt.m_delivery_status != check::CHK_DISCARD
                && pred(rcpt.domainType);
        });
    return {rcpts.begin(), rcpts.end()}; 
}

void TAsyncDeliveryImpl::Dsn(dsn::composer::type_t type, TErrorCode ec, std::string answer) {
    auto& envelope = DlvRequest.Envelope;
    const auto&  sessionId = DlvRequest.SessionId;
    reenter(DsnCoroutine) {
        try {
            DsnEnvelope = MakeDsnEnvelope(type);
            if (!DsnEnvelope) {
                if (type == dsn::composer::failure) {
                    std::tie(ec, answer) = CalculateResponse(envelope, false);
                }
                return boost::asio::post(Io,
                    std::bind(&TAsyncDeliveryImpl::Delivery, shared_from_this(), std::move(ec), std::move(answer)));
                
            }
        } catch (const std::exception& exc) {
            NWLOG_L_EXC(notice, "DSN", "Exception occurred", exc);
            if (type == dsn::composer::failure) {
                std::tie(ec, answer) = CalculateResponse(envelope, false);
            }
            return boost::asio::post(Io,
                std::bind(&TAsyncDeliveryImpl::Delivery, shared_from_this(), std::move(ec), std::move(answer)));
            
        }
        if (gconfig->dsn.sign) {
            yield {
                dkim_sign::input inp(
                    gconfig->dsn.composer.origin,
                    DsnEnvelope->header_storage_,
                    TBufferRange(DsnEnvelope->orig_message_body_beg_, DsnEnvelope->orig_message_.cend()),
                    sessionId,
                    envelope->m_id,
                    [self = shared_from_this(), context = Context, sessionId, envelope](const auto& msg){
                        NWLOG_CTX(notice, context, "DKIM", msg);
                    }
                );
                auto dkimHandler =
                    [this, self = shared_from_this(), type]
                    (const std::string& error, const dkim_sign::Output& out) {
                        const std::string where {"DKIM-DSN"};
                        if (!error.empty()) {
                            NWLOG_CTX(notice, Context, where, error);
                        } else {
                            NWLOG_CTX(notice, Context, where, "added a signature for " + out.identity);
                            NUtil::AppendToSegment(DsnEnvelope->added_headers_, out.header);
                        }
                        boost::asio::post(Io, std::bind(&TAsyncDeliveryImpl::Dsn, self, type, TErrorCode{}, ""));
                    };
                AsyncDkimSign(inp, dkimHandler);
            }
        }
        {
            std::string addedHeaders = std::string{DsnEnvelope->added_headers_.cbegin(), DsnEnvelope->added_headers_.cend()};
            std::string origMessage = std::string{DsnEnvelope->orig_message_.cbegin(), DsnEnvelope->orig_message_.cend()};

            DsnEnvelope->altered_message_ = NUtil::JoinStrings(addedHeaders, origMessage);
        }
        yield SmtpClient->Start(
            std::bind(&TAsyncDeliveryImpl::Dsn, shared_from_this(), type, p::_1, ""),
            DsnEnvelope,
            sessionId,
            gconfig->dsn.relay,
            type == dsn::composer::failure ? "DSN-FAIL" : "DSN-SUCC",
            gconfig->delivery.fallback.timeouts
        );
        // If failure DSN was not accepted by our DSN relay we report an error
        // in SMTP session.
        if (type == dsn::composer::failure && ec) {
            std::tie(ec, answer) = CalculateResponse(envelope, false);
        }
        boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::Delivery, shared_from_this(), std::move(ec), std::move(answer)));
    }
}

#include <boost/asio/unyield.hpp>

envelope_ptr TAsyncDeliveryImpl::MakeDsnEnvelope(dsn::composer::type_t type) {
    auto& env = DlvRequest.Envelope;
    dsn::composer composer(gconfig->dsn.composer);
    if (!gconfig->mydomain.empty()) {
        composer.my_domain(gconfig->mydomain);
    }
    std::vector<dsn::rcpt> rcpts;
    for (const auto& rcpt : env->m_rcpts) {
        if (rcpt.m_delivery_status != check::CHK_ACCEPT
            && rcpt.m_delivery_status != check::CHK_REJECT
            && rcpt.m_delivery_status != check::CHK_TEMPFAIL) {
            continue;
        }
        rcpts.push_back({
            rcpt.m_name,
            rcpt.m_remote_answer,
            rcpt.m_notify_mode,
            rcpt.m_delivery_status == check::CHK_ACCEPT ? dsn::rcpt::success : dsn::rcpt::failure});
    }
    const auto [success, msg] = composer.compose(type, env->m_id, env->m_original_id, env->m_sender, rcpts);
    if (!success) {
        return {};
    }
    envelope_ptr dsn(new envelope());
    dsn->add_recipient(
        /*rcpt=           */ env->m_sender,
        /*suid=           */ 0,
        /*is_alias=       */ true,
        /*uid=            */ "",
        /*ml_uid=         */ "",
        /*default_email=  */ "",
        /*country_code=   */ "",
        /*domainType=     */ DomainType::UNKNOWN,
        /*notify_mode=    */ dsn::Options::NEVER);
    dsn->orig_message_ = NUtil::MakeSegment(msg);
    auto [headers, body] = ParseMessage(dsn->orig_message_);
    if (headers.Empty()) {
        dsn->orig_message_body_beg_ = dsn->orig_message_.cbegin();
    } else {
        dsn->orig_message_body_beg_ = body.begin();
    }
    return dsn;
}

void TAsyncDeliveryImpl::Local() {
    SmtpPoint point = gconfig->delivery.local.addr;
    SmtpTimeouts timeouts = gconfig->delivery.local.timeouts;
    SmtpClient->Start(
        std::bind(&TAsyncDeliveryImpl::Delivery, shared_from_this(), p::_1, ""),
        DlvRequest.Envelope,
        CurrentRcpts,
        DlvRequest.SessionId,
        DlvRequest.Envelope->m_sender,
        point,
        "LOCAL",
        timeouts,
        gconfig->delivery.localTargeting
    );
}

void TAsyncDeliveryImpl::Back() {
    SmtpClient->Start(
        std::bind(&TAsyncDeliveryImpl::Delivery, shared_from_this(), p::_1, ""),
        DlvRequest.Envelope,
        CurrentRcpts,
        DlvRequest.SessionId,
        DlvRequest.Envelope->m_sender,
        gconfig->delivery.fallback.addr,
        "SMTP",
        gconfig->delivery.fallback.timeouts,
        gconfig->delivery.fallbackTargeting
    );
}

void TAsyncDeliveryImpl::External(const SmtpPoint& point, const SmtpTimeouts& timeouts) {
    const auto& sender =
        (gconfig->msgOpts.rewriteSenderFromHeader
            && !SenderDependentRelay.host.empty()
            && !DlvRequest.Envelope->address_sender_from_mailbody.empty())
        ? DlvRequest.Envelope->address_sender_from_mailbody : DlvRequest.Envelope->m_sender;

    SmtpClient->Start(
        std::bind(&TAsyncDeliveryImpl::Delivery, shared_from_this(), p::_1, ""),
        DlvRequest.Envelope,
        CurrentRcpts,
        DlvRequest.SessionId,
        sender,
        point,
        "SMTP-RELAY",
        timeouts,
        std::nullopt,
        AuthLogin,
        AuthPassword,
        AuthOauth
    );
}

} // namespace NNwSmtp::NAsyncDlv
