#pragma once

#include <boost/system/error_code.hpp>

namespace NNwSmtp::NAsyncDlv {

enum class EError {
    Ok,
    TooManyHeadersReject,
    DkimSignTempFail,
    AvirDiscarded,
    AvirRejected,
    RpopAuthTempFail,
    RpopAuthRejected,
    DeliveryTempFail,
    DeliveryRejected,
    SenderIdRejected,
    RouteTempFail,
    MaliciousDiscarded,
    MaliciousRejected,
    SpamDiscarded,
    SpamRejected,
    SoTempFail,
    RfcFailRejected,
    UrlRblRejected,
    BadKarmaRejected,
    MailLimitsRejected,
    PddAdminKarmaRejected,
    BouncesRejected,
    SpamComplRejected,
    RelayNotSet,
    CycleDetected
};

} // namespace NNwSmtp::NAsyncDlv

namespace boost::system {

template <>
struct is_error_code_enum<NNwSmtp::NAsyncDlv::EError> : std::true_type {};

} // namespace boost::system

namespace NNwSmtp::NAsyncDlv {

class TErrorCategory final : public boost::system::error_category {

public:
    const char* name() const noexcept override {
        return "nwsmtp_async_delivery";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::Ok:
                return "Ok";
            case EError::TooManyHeadersReject:
                return "Too many headers";
            case EError::DkimSignTempFail:
                return "Dkim signing process temporarily failed";
            case EError::AvirDiscarded:
                return "Virus email discarded";
            case EError::AvirRejected:
                return "Virus email rejected";
            case EError::RpopAuthTempFail:
                return "Rpop auth temporarily failed";
            case EError::RpopAuthRejected:
                return "Rpop auth rejected";
            case EError::DeliveryTempFail:
                return "Delivery temporarily failed";
            case EError::DeliveryRejected:
                return "Delivery rejected";
            case EError::SenderIdRejected:
                return "Sender id rejected";
            case EError::RouteTempFail:
                return "Route failed";
            case EError::MaliciousRejected:
                return "Malicious spam email rejected";
            case EError::MaliciousDiscarded:
                return "Malicious spam email discarded";
            case EError::SpamRejected:
                return "Spam email rejected";
            case EError::SpamDiscarded:
                return "Spam email discarded";
            case EError::SoTempFail:
                return "So temporarily failed";
            case EError::RfcFailRejected:
                return "Сritical violation of the RFC standard";
            case EError::UrlRblRejected:
                return "Surbl url RBL";
            case EError::BadKarmaRejected:
                return "The sender's bad karma";
            case EError::MailLimitsRejected:
                return "Daily dispatch limits";
            case EError::PddAdminKarmaRejected:
                return "Bad karma of the domain owner";
            case EError::BouncesRejected:
                return "External mail does not accept sender/domain emails";
            case EError::SpamComplRejected:
                return "Complaints about the sender";
            case EError::RelayNotSet:
                return "Relay not set";
            case EError::CycleDetected:
                return "cycle detected";
        }
        return "Unknown error code: " + std::to_string(value);
    }
};

inline const TErrorCategory& getErrorCategory() {
    static TErrorCategory errorCategory;
    return errorCategory;
}

inline auto make_error_code(EError ec) {
    return boost::system::error_code(static_cast<int>(ec), getErrorCategory());
}

} // namespace NNwSmtp::NAsyncDlv
