#include "async_impl.h"
#include "route_calculator.h"
#include "so.h"
#include "utils.h"

#include <mail/nwsmtp/src/control_from/control_from.h>
#include <mail/nwsmtp/src/dkim/dkim_module.h>
#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/options.h>
#include <mail/nwsmtp/src/so/utils.h>
#include <mail/nwsmtp/src/public_suffix_list.h>
#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/src/yarm/error_code.h>

#include <mail/sova/include/nwsmtp/avir_client.h>

#include <boost/algorithm/string.hpp>
#include <boost/asio.hpp>
#include <boost/format.hpp>
#include <boost/range/algorithm/for_each.hpp>

#include <functional>
#include <memory>

namespace NNwSmtp::NAsyncDlv {

namespace ph = std::placeholders;

#include <boost/asio/yield.hpp>

void TAsyncDeliveryImpl::PrepareDelivery(TErrorCode ec, std::string answer) {
    const auto& envelope = DlvRequest.Envelope;

    reenter (PrepareDeliveryCoroutine) {
        HasDkimHeaders = envelope->header_storage_->Contains("dkim-signature");

        if (gconfig->msgOpts.ControlFromOpts.Policy != NControlFrom::EPolicy::None) {
            yield ControlFrom();
            if (ec) {
                return boost::asio::post(Io,
                    std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
            }
        }

        if (HasDkimHeaders &&
            (DkimMode == DkimOptions::verify || DkimMode == DkimOptions::signverify)
        ) {
            yield DkimCheck();
            if (ec) {
                return boost::asio::post(Io,
                    std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
            }
        }

        if (gconfig->dmarc.use && !FromDomain.empty()) {
            yield DmarcCheck();
            if (ec) {
                return boost::asio::post(Io,
                    std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
            }
        }

        if (!HasDkimHeaders &&
            (DkimMode == DkimOptions::sign || DkimMode == DkimOptions::signverify)
        ) {
            yield DkimSign();
            if (ec) {
                return boost::asio::post(Io,
                    std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
            }
        }

        AddAuthResultsHeader();

        if (gconfig->soOpts.Check) {
            if (SkipSoCheck) {
                NWLOG_L(notice, "RECV", "x-yandex-spam=" + NSO::ResolutionToString(SpamHeaderStatus));
                if (SpamHeaderStatus == NSO::EResolution::SO_RESOLUTION_REJECT) {
                    if (gconfig->soOpts.ActionMalicious == NSO::ESpamPolicy::DISCARD) {
                        return boost::asio::post(Io,
                            std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), EError::MaliciousDiscarded, ""));
                    }
                    return boost::asio::post(Io,
                        std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), EError::MaliciousRejected, "554 5.7.1 " + gconfig->soOpts.ReplyTextMalicious));
                }
            } else {
                yield SoCheck();
                if (ec) {
                    return boost::asio::post(Io,
                        std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
                }
            }
        }

        if (gconfig->avir.check && SkipAvirCheck) {
            NWLOG_L(notice, "RECV", "x-yandex-avir=" + std::to_string(VirusHeaderStatus));

            if (VirusHeaderStatus == avir_client::infected) {
                std::tie(ec, answer) = ProcessAvirStatus(avir_client::infected);
                return boost::asio::post(Io,
                    std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
            }
            if (VirusHeaderStatus != avir_client::clean && gconfig->avir.check && envelope->orig_message_size_ > 0) {
                yield AvirCheckData();
                if (ec) {
                    return boost::asio::post(Io,
                        std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
                }
            }
        } else if (gconfig->avir.check && envelope->orig_message_size_ > 0) {
            yield AvirCheckData();
            if (ec) {
                return boost::asio::post(Io,
                    std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
            }
        }

        if (NUtil::IsRouting()) {
            yield Route();
            if (ec) {
                return boost::asio::post(Io,
                    std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
            }
        }

        if (!gconfig->delivery.senderDependent.relaysMap.empty()) {
            SenderDependentRelay = CheckSenderDependentRelayMap(envelope, DlvRequest.SmtpFrom);
        }

        if (gconfig->yarm.useRpopAuth && !SenderDependentRelay.host.empty()) {
            if (envelope->m_sender_uid.empty()) {
                return boost::asio::post(Io,
                    std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), EError::SenderIdRejected, "554 5.7.0 Sender Id required"));
            }
            yield GetRpopAuth();
            if (ec) {
                return boost::asio::post(Io,
                    std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), std::move(ec), std::move(answer)));
            }
        }

        if (!envelope->foreign_mx.empty()) {
            if (
                auto mxcode = gconfig->delivery.mxCodeMap.find(envelope->foreign_mx);
                mxcode != gconfig->delivery.mxCodeMap.end() && !mxcode->second.host_name_.empty()
            ) {
                ForeignMxRelay = NNwSmtp::detail::remote_point_to_smtp(mxcode->second);
            }
        }

        PrepareAlteredMessage();

        boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), TErrorCode{}, ""));
    }
}

#include <boost/asio/unyield.hpp>

void TAsyncDeliveryImpl::ControlFrom() {
    std::make_shared<NControlFrom::control_from>(Context, Io)->start(
        {DlvRequest.Envelope, DlvRequest.SessionId, DlvRequest.Ip.to_string()},
        std::bind(&TAsyncDeliveryImpl::HandleControlFrom,  shared_from_this(), ph::_1, ph::_2)
    );
}

void TAsyncDeliveryImpl::HandleControlFrom(TErrorCode ec, NControlFrom::TResponse response) {
    using NControlFrom::EError;

    if (ec == EError::Accept || ec == EError::WhiteList ||
            gconfig->msgOpts.ControlFromOpts.Policy != NControlFrom::EPolicy::Reject) {
        return  boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), TErrorCode{}, ""));
    }
    boost::asio::post(Io,
        std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), std::move(ec), std::move(response.Remote)));
}

void TAsyncDeliveryImpl::DkimCheck() {
    const auto& envelope = DlvRequest.Envelope;
    dkim_check::input input(
        FromDomain,
        envelope->header_storage_,
        TBufferRange{envelope->orig_message_body_beg_, envelope->orig_message_.cend()},
        [sessionId = DlvRequest.SessionId, envelope = envelope, context = Context](const auto& message){
            NWLOG_CTX(notice, context, "DKIM", message);
        }
    );
    AsyncDkimCheck(
        input,
        gconfig->aresOpts.resolverOptions,
        [this, self = shared_from_this()] (dkim_check::Output output) {
            boost::asio::post(Io, std::bind(&TAsyncDeliveryImpl::HandleDkimCheck,
                self, std::move(output)));
        }
    );
}

void TAsyncDeliveryImpl::HandleDkimCheck(dkim_check::Output output) {
    DkimStatus = output.status;
    DkimIdentity = std::move(output.identity);
    DkimDomain = std::move(output.domain);
    if (gconfig->dmarc.use) {
        DkimDomains = std::move(output.domainList);
    }
    boost::asio::post(Io,
        std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), TErrorCode{}, ""));
}

void TAsyncDeliveryImpl::DmarcCheck() {
    DmarcChecker = std::make_shared<dmarc::checker>(Io, gconfig->aresOpts.resolverOptions);
    DmarcChecker->start(
        FromDomain,
        [this, self = shared_from_this()] (std::optional<dmarc::record> record) {
            boost::asio::post(Io, std::bind(&TAsyncDeliveryImpl::HandleDmarcCheck,
                self, std::move(record)));
        },
        Context
    );
}

void TAsyncDeliveryImpl::HandleDmarcCheck(std::optional<dmarc::record> record) {
    if (record) {
        bool dkimAligned = false;
        bool spfAligned = false;
        std::string additionalInfo;
        std::string fromTld;
        if (!record->m_strict_dkim || !record->m_strict_spf) {
            fromTld = g_public_suffixes.find_tld(FromDomain);
        }
        if (DkimStatus == dkim_check::pass) {
            for (const auto& domain : DkimDomains) {
                if (record->m_strict_dkim) {
                    dkimAligned = boost::iequals(FromDomain, domain);
                } else
                    dkimAligned = boost::iequals(fromTld, g_public_suffixes.find_tld(domain));
                if (dkimAligned) {
                    break;
                }
            }
            additionalInfo += str(boost::format(", dkim_aligned=%1%") % dkimAligned);
        }

        const auto& spfResult = DlvRequest.SpfResult;
        const auto& spfExpl = DlvRequest.SpfExpl;

        std::string::size_type at = DlvRequest.SmtpFrom.find('@');
        std::string envelopeDomain = (at != std::string::npos ? DlvRequest.SmtpFrom.substr(at + 1) : "");
        if (spfResult && spfExpl && spfResult.value() == "pass") {
            if (record->m_strict_spf) {
                spfAligned = boost::iequals(FromDomain, envelopeDomain);
            } else {
                spfAligned = boost::iequals(fromTld, g_public_suffixes.find_tld(envelopeDomain));
            }
            additionalInfo += str(boost::format(", spf_aligned=%1%") % spfAligned);
        }

        if (record->m_subdomain_policy != dmarc::record::UNKNOWN && record->m_domain != FromDomain) {
            DmarcPolicy = record->m_subdomain_policy;
        } else {
            DmarcPolicy = record->m_policy;
        }
        if (dkimAligned || spfAligned) {
            DmarcPolicy = dmarc::record::NONE;
        }
        DmarcDomain = record->m_domain;
        NWLOG_L(notice, "DMARC", "domain='" + record->m_domain + "', policy=" + dmarc::convert_policy(
            DmarcPolicy) + ", stat=ok" + additionalInfo);
    }

    DmarcChecker.reset();
    boost::asio::post(Io,
        std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), TErrorCode{}, ""));
}

void TAsyncDeliveryImpl::DkimSign() {
    const auto& envelope = DlvRequest.Envelope;
    const auto& sessionId = DlvRequest.SessionId;
    dkim_sign::input input(
        DlvRequest.SmtpFrom,
        envelope->header_storage_,
        TBufferRange(envelope->orig_message_body_beg_, envelope->orig_message_.cend()),
        sessionId,
        envelope->m_id,
        [sessionId = sessionId, envelope = envelope, context = Context] (const auto& message){
            NWLOG_CTX(notice, context, "DKIM", message);
        }
    );
    AsyncDkimSign(
        input,
        [this, self = shared_from_this()] (auto ec, auto output) {
            boost::asio::post(Io, std::bind(&TAsyncDeliveryImpl::HandleDkimSign,
                self, std::move(ec), std::move(output)));
        }
    );
}

void TAsyncDeliveryImpl::HandleDkimSign(std::string error, dkim_sign::Output output) {
    const auto& envelope = DlvRequest.Envelope;
    const std::string where {"DKIM"};
    if (!error.empty()) {
        NWLOG_CTX(notice, Context, where, error);

        if(output.hasError && !SkipDkimSignIfError) {
            return boost::asio::post(Io,
                std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), EError::DkimSignTempFail, "451 4.3.0 Try again later"));
        }
    }

    if (DkimMode == DkimOptions::signverify) {
        DkimStatus = dkim_check::pass;
        DkimIdentity = output.identity;
    }

    NWLOG_CTX(notice, Context, where, "added a signature for " + output.identity);

    NUtil::AppendToSegment(envelope->added_headers_, output.header);

    boost::asio::post(Io,
        std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), TErrorCode{}, ""));
}

void TAsyncDeliveryImpl::SoCheck() {
    auto soRequest = BuildSoRequest(
        DlvRequest,
        AuthResultsHeaderValue,
        DmarcPolicy,
        DmarcDomain,
        gconfig->soOpts
    );

    SoClient->Check(
        Context,
        std::move(soRequest),
        std::bind(&TAsyncDeliveryImpl::HandleSoCheck, shared_from_this(), ph::_1, ph::_2)
    );
}

void TAsyncDeliveryImpl::HandleSoCheck(TErrorCode ec, NSO::TResponse response) {
    if (ec) {
        return boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), EError::SoTempFail, "451 4.3.0 Try again later"));
    }
    std::string answer;
    tie(ec, answer) = MakeSoResult(
        response,
        DlvRequest.Envelope,
        DmarcDomain,
        DmarcPolicy,
        gconfig->soOpts,
        SoCaptcha
    );
    LogSoStatus(Context, DlvRequest.Envelope);
    boost::asio::post(Io,
        std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), std::move(ec), std::move(answer)));
}

void TAsyncDeliveryImpl::AvirCheckData() {
    AvirCheckClient->Check(
        Context,
        DlvRequest.Envelope->orig_message_,
        std::bind(&TAsyncDeliveryImpl::HandleAvirCheck, shared_from_this(), ph::_1, ph::_2)
    );
}

void TAsyncDeliveryImpl::HandleAvirCheck(TErrorCode, NAvir::TStatus status) {
    auto&& [ec, answer] = ProcessAvirStatus(status);

    DlvRequest.Envelope->avirStatus = status;

    if (ec) {
        boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), std::move(ec), std::move(answer)));
    } else {
        boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), TErrorCode{}, ""));
    }
}

void TAsyncDeliveryImpl::AddAuthResultsHeader() {
    const auto& envelope = DlvRequest.Envelope;
    const auto& spfResult = DlvRequest.SpfResult;
    const auto& spfExpl = DlvRequest.SpfExpl;
    bool hasDkim = DkimStatus != dkim_check::none;
    bool hasSpf = spfResult && spfExpl;

    if (hasDkim || hasSpf) {
        std::string additionalHeader;
        std::string dkimIdentity;
        if (hasDkim && !DkimIdentity.empty()) {
            dkimIdentity = (boost::format(" header.i=%1%") % DkimIdentity).str();
        }
        if (hasDkim && hasSpf) {
            additionalHeader = (boost::format("%1%; spf=%2% (%3%) smtp.mail=%4%; dkim=%5%%6%")
                % boost::asio::ip::host_name()
                % spfResult.value() % spfExpl.value() % DlvRequest.SmtpFrom
                % dkim_check::status(DkimStatus) % dkimIdentity).str();
        } else if (hasSpf) {
            additionalHeader = (boost::format("%1%; spf=%2% (%3%) smtp.mail=%4%")
                % boost::asio::ip::host_name()
                % spfResult.value() % spfExpl.value() % DlvRequest.SmtpFrom).str();
        } else {
            additionalHeader = (boost::format("%1%; dkim=%2%%3%")
                % boost::asio::ip::host_name()
                % dkim_check::status(DkimStatus) % dkimIdentity).str();
        }

        AuthResultsHeaderValue = additionalHeader;

        additionalHeader = (boost::format("Authentication-Results: %1%\r\n") % additionalHeader).str();
        NWLOG_L(notice, "Authentication-Results", additionalHeader);

        NUtil::AppendToSegment(envelope->added_headers_, additionalHeader);
    }
}

void TAsyncDeliveryImpl::GetRpopAuth() {
    const auto& envelope = DlvRequest.Envelope;
    const auto& sender = (gconfig->msgOpts.rewriteSenderFromHeader && !envelope->address_sender_from_mailbody.empty())
        ? envelope->address_sender_from_mailbody
        : envelope->m_sender;

    YarmClient->AsyncRun(
        Context,
        NYarm::TRequest {
            .Uid = envelope->m_sender_uid,
            .Email = sender,
        },
        [this, self = shared_from_this()](TErrorCode ec, NYarm::TResponse response) {
            boost::asio::post(Io, std::bind(&TAsyncDeliveryImpl::HandleRpopAuth, self, std::move(ec), std::move(response)));
        }
    );
}

void TAsyncDeliveryImpl::HandleRpopAuth(TErrorCode ec, NYarm::TResponse response) {
    if (ec) {
        std::string answer;
        if (ec == NYarm::EError::NotFound) {
            ec = EError::RpopAuthRejected;
            answer = "554 5.7.0 Failed to authorize the sender";
        } else {
            ec = EError::RpopAuthTempFail;
            answer = "451 4.7.1 Sorry, the service is currently unavailable. Please come back later.";
        }
        boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), std::move(ec), std::move(answer)));
    } else {
        AuthLogin = std::move(response.Login);
        AuthPassword = std::move(response.PasswordOrToken);
        AuthOauth = response.IsOauth;
        boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), TErrorCode{}, ""));
    }
}

void TAsyncDeliveryImpl::Route() {
    std::make_shared<RouteCalculator>(
        std::bind(&TAsyncDeliveryImpl::HandleRoute, shared_from_this(), ph::_1),
        Context,
        Router,
        DlvRequest.Envelope->m_rcpts.begin(),
        DlvRequest.Envelope->m_rcpts.end(),
        Io
    )->Run();
}

void TAsyncDeliveryImpl::HandleRoute(TErrorCode ec) {
    if (ec) {
        NWLOG_L_EC(error, "ROUTE", "Error occurred", ec);
        boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), EError::RouteTempFail, "451 4.7.1 Service unavailable - try again later"));  
    } else {
        boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::PrepareDelivery, shared_from_this(), TErrorCode{}, ""));
    }
}

void TAsyncDeliveryImpl::PrepareAlteredMessage() {
    auto& envelope = DlvRequest.Envelope;
    auto& alteredMessage = envelope->altered_message_;
    auto& addedHeaders = envelope->added_headers_;

    if (!envelope->xyandex_hint_header_value_.empty()) {
        std::string encodedHint;
        encode_to_base64(envelope->xyandex_hint_header_value_, encodedHint);
        auto header = str(boost::format("X-Yandex-Hint: %1%\r\n") % encodedHint);
        NUtil::AppendToSegment(addedHeaders, header);
    }

    boost::range::for_each(envelope->xyandex_hint_header_value_personal_,
        [&](const auto& value) {
            std::string encodedHint;
            encode_to_base64(value, encodedHint);
            auto header = str(boost::format("X-Yandex-Hint: %1%\r\n") % encodedHint);
            NUtil::AppendToSegment(addedHeaders, header);
        }
    );

    auto headers = boost::join(
        envelope->filtered_header_storage_->GetAllHeaders() | boost::adaptors::transformed(
            [](const auto& value) { 
                auto header = value.GetWhole();
                return std::string{header.begin(), header.end()}; 
            }), 
        "\r\n" ) + "\r\n\r\n";

    auto addedHeadersStr = std::string{addedHeaders.cbegin(), addedHeaders.cend()};
    auto origMessage = GetMessageBody(envelope);
    alteredMessage = NUtil::JoinStrings(addedHeadersStr, headers, origMessage);
}

} // namespace NwSmtp::NAsyncDlv
