#include "async_impl.h"
#include "utils.h"

#include <mail/nwsmtp/src/delivery/async/read_headers.h>
#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/src/xyandexhint/xyandexhint.h>

#include <boost/algorithm/string.hpp>
#include <boost/asio.hpp>
#include <boost/format.hpp>

#include <functional>
#include <vector>

namespace NNwSmtp::NAsyncDlv {

void TAsyncDeliveryImpl::PrepareHeaders() {
    const auto& envelope = DlvRequest.Envelope;
    const auto& sessionId = DlvRequest.SessionId;

    auto&& parseResult = ProcessHeaders(envelope, Recognizer);

    const auto headersSize = parseResult.headers.size();
    if (headersSize > gconfig->msgOpts.headersCountLimit) {
        NWLOG_L(notice, "", "too many headers, count=" + std::to_string(headersSize));
        return boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), EError::TooManyHeadersReject, "550 5.3.4 Too many headers"));
    }

    auto& addedHeaders = envelope->added_headers_;

    NUtil::AppendToSegment(addedHeaders, envelope->received_headers_);
    if (gconfig->deliveryToSenderControl.use) {
        std::string hdr{"X-Yandex-Internal: 1\r\n"};
        NUtil::AppendToSegment(addedHeaders, hdr);
    }

    envelope->orig_message_body_beg_ = parseResult.body_iter;

    auto& parameters = parseResult.parameters;
    FromDomain = std::move(parameters.from_domain);
    SpamHeaderStatus = parameters.spam_status;
    VirusHeaderStatus = parameters.virus_status;
    SoCaptcha = parameters.captcha;

    MailerHdrName = std::move(parameters.mailer_hdr_name);
    Mailer = std::move(parameters.mailer);
    squash_consecutive_spaces(Mailer);

    if (gconfig->decyclerOpts.reject && parameters.decycler_counter >= gconfig->decyclerOpts.ttl) {
        NWLOG_L(notice, "", "too many hops, count=" + std::to_string(parameters.decycler_counter));
        return boost::asio::post(Io,
            std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), EError::CycleDetected, "554 5.4.0 Error: too many hops"));
    }

    if (request_id_valid(parameters.request_id)) {
        RequestId = std::move(parameters.request_id);
    }

    if (!Mailer.empty()) {
        NWLOG_L(notice, "", "mailer='" + Mailer + "', header='" + MailerHdrName + "'");
    } else {
        NWLOG_L(notice, "", "mailer=''");
    }

    bool hasPersonalHint = false;
    for (const auto& parsed_header : parseResult.headers) {
        const auto& value = parsed_header.value;
        const auto& name = parsed_header.name;
        if (gconfig->deliveryToSenderControl.use &&
            boost::iequals(name, "x-yandex-hint")
        ) {
            NHint::TParamsContainer params;
            NHint::ParseBase64({value.begin(), value.end()}, params);
            TXYandexHint<NHint::TEmail, NHint::TCopyToInbox, NHint::TSaveToSent> hint;
            hint.Fill(params);
            if (boost::iequals(hint.GetTEmail().value, envelope->m_sender)) {
                if (params.find("save_to_sent") == params.end()) {
                    params.emplace("save_to_sent", "0");
                }

                params.erase("copy_to_inbox");
                if (envelope->is_sender_in_rcpts_) {
                    params.emplace("copy_to_inbox", "1");
                } else {
                    params.emplace("copy_to_inbox", "0");
                }

                if (params.find("filters") == params.end()) {
                    params.emplace("filters", "0");
                }
                if (params.find("skip_loop_prevention") == params.end()) {
                    params.emplace("skip_loop_prevention", "1");
                }
                if (params.find("notify") == params.end()) {
                    params.emplace("notify", "0");
                }

                hasPersonalHint = true;
                std::string hdr{MakeHintFromParams(params)};
                NUtil::AppendToSegment(addedHeaders, hdr);
                continue;
            }
        } else if (gconfig->decyclerOpts.updateHeader && boost::iequals(name, "x-yandex-fwd")) {
            std::string hdr{str(boost::format("X-Yandex-Fwd: %1%\r\n") % std::to_string(parameters.decycler_counter + 1))};
            NUtil::AppendToSegment(addedHeaders, hdr);
            NWLOG_L(notice, "DECYCLER", "x-yandex-fwd=" + std::to_string(parameters.decycler_counter));
            continue;
        }

        envelope->filtered_header_storage_->Add(name, value);
    }

    if (gconfig->deliveryToSenderControl.use
        && !envelope->m_sender_uid.empty()
        && !hasPersonalHint
    ) {
        NHint::TParamsContainer params;
        params.emplace("email", envelope->m_sender);
        params.emplace("save_to_sent", "0");
        params.emplace("filters", "0");
        params.emplace("skip_loop_prevention", "1");
        params.emplace("notify", "0");
        if (envelope->is_sender_in_rcpts_) {
            params.emplace("copy_to_inbox", "1");
        } else {
            params.emplace("copy_to_inbox", "0");
        }
        std::string hdr{MakeHintFromParams(params)};
        NUtil::AppendToSegment(addedHeaders, hdr);
    }

    if (!envelope->header_storage_->Contains("message-id")) {
        std::string timestamp = GetTimestamp();

        // format: <20100406110540.C671D18D007F@mxback1.mail.yandex.net>
        std::string messageId = str(boost::format("<%1%.%2%@%3%>")
            % timestamp % envelope->m_id % boost::asio::ip::host_name());

        std::string hdr{str(boost::format("Message-Id: %1%\r\n") % messageId)};
        NUtil::AppendToSegment(addedHeaders, hdr);

        NWLOG_L(notice, "RECV", "message-id=" + messageId);

        envelope->message_id_ = messageId;
        envelope->added_message_id_ = true;
    } else {
        NWLOG_L(notice, "RECV", "message-id=" + envelope->message_id_);
    }

    if (!envelope->header_storage_->Contains("date")) {
        std::string hdr{str(boost::format("Date: %1%\r\n") % NUtil::GetRfc822DateNow())};
        NUtil::AppendToSegment(addedHeaders, hdr);
    }
    if (!envelope->header_storage_->Contains("from")) {
        std::string hdr{"From: MAILER-DAEMON\r\n"};
        NUtil::AppendToSegment(addedHeaders, hdr);
    }
    if (!envelope->header_storage_->Contains("to")) {
        std::string hdr{"To: undisclosed-recipients:;\r\n"};
        NUtil::AppendToSegment(addedHeaders, hdr);
    }

    if (gconfig->msgOpts.addReturnPath && !DlvRequest.SmtpFrom.empty()) {
        std::string hdr{str(boost::format("Return-Path: %1%\r\n") % DlvRequest.SmtpFrom)};
        NUtil::AppendToSegment(addedHeaders, hdr);
    }

    if (gconfig->msgOpts.markMLMessage && envelope->is_mailing_list_) {
        std::string hdr{"Precedence: bulk\r\n"};
        hdr += (boost::format("List-Unsubscribe: <https://"
            "ml.yandex-team.ru/lists/%1%/unsubscribe-click>\r\n")
            % envelope->ml_address_).str();
        TXYandexHint<NHint::TEmail, NHint::TAllowAutoReply> hint(
            NHint::TEmail{envelope->ml_address_}, NHint::TAllowAutoReply{true});
        hdr += hint.Str();

        NUtil::AppendToSegment(addedHeaders, hdr);
    }

    if ((gconfig->msgOpts.trustHeaders.count("x-yandex-spam") &&
            envelope->header_storage_->Contains("x-yandex-spam")) ||
        (gconfig->msgOpts.trustHeaders.count("x-spam-flag") &&
            envelope->header_storage_->Contains("x-spam-flag"))
    ) {
        SkipSoCheck = true;
    }

    if (gconfig->decyclerOpts.updateHeader && !envelope->header_storage_->Contains("x-yandex-fwd")) {
        NWLOG_L(notice, "DECYCLER", "x-yandex-fwd=1");
        std::string hdr{str(boost::format("X-Yandex-Fwd: 1\r\n"))};

        NUtil::AppendToSegment(addedHeaders, hdr);
    }

    if (gconfig->msgOpts.trustHeaders.count("x-yandex-avir") &&
        envelope->header_storage_->Contains("x-yandex-avir")
    ) {
        SkipAvirCheck = true;
    }

    const auto xYHintParameters = GetNonpersonalHintsParameters(
        envelope->header_storage_->GetHeaders("x-yandex-hint")
    );

    if (!Contains(xYHintParameters, "session_id")) {
        envelope->xyandex_hint_header_value_.append(
            str(boost::format("session_id=%1%-%2%\n") % sessionId % envelope->m_id)
        );
    }

    if (!DlvRequest.IsSSLNone) {
        envelope->xyandex_hint_header_value_.append("label=symbol:encrypted_label\n");
    }

    if (!Contains(xYHintParameters, "ipfrom")) {
        envelope->xyandex_hint_header_value_.append(
            str(boost::format("ipfrom=%1%\n") % DlvRequest.Ip.to_string())
        );
    }

    boost::asio::post(Io,
        std::bind(&TAsyncDeliveryImpl::Run, shared_from_this(), TErrorCode{}, ""));
}

} // namespace NwSmtp::NAsyncDlv
