#include "read_headers.h"
#include "header_storage.h"
#include "utils.h"

#include <mail/nwsmtp/src/xyandexhint/xyandexhint.h>
#include <mail/nwsmtp/src/dkim/adkim.h>
#include <mail/nwsmtp/src/so/utils.h>
#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/xyandexhint/xyandexhint.h>
#include <boost/format.hpp>
#include <boost/lexical_cast/try_lexical_convert.hpp>

#include <optional>
#include <vector>

namespace NNwSmtp {

const std::set<std::string> MAILER_HEADER_NAMES = {"x-mailer", "user-agent", "x-mimeole", "x-ms-tnef-correlator"};

NSO::EResolution FromSpamFLag(const std::string& value) {
    if (boost::iequals(value, "NO")) {
        return NSO::EResolution::SO_RESOLUTION_ACCEPT;
    }
    if (boost::iequals(value, "YES")) {
        return NSO::EResolution::SO_RESOLUTION_SPAM;
    }
    return NSO::EResolution::SO_RESOLUTION_SKIP;
}

parse_header_result_t ProcessHeaders (
    const envelope_ptr& envelope,
    TRecognizerPtr recognizer
) {
    parse_header_result_t result;

    auto [headers, body] = ParseMessage(envelope->orig_message_);
    if (!headers.Empty()) {
        result.body_iter = body.begin();
    } else {
        result.body_iter = envelope->orig_message_.begin();
    }

    boost::range::for_each(headers.GetAllHeaders(), [&](const auto& hdr) {
        auto name = hdr.Name;
        auto value = hdr.Value;

        std::string lname; // lower-cased header name
        size_t name_sz = name.size();
        lname.reserve(name_sz);
        std::transform(name.begin(), name.end(), back_inserter(lname), ::tolower);

        envelope->header_storage_->Add(name, value);

        if (lname == "message-id") {
            envelope->message_id_.assign(value.begin(), value.end());
        } else if (lname == "from") {
            if (!envelope->header_from_) {
                envelope->header_from_ = value;
            }

            std::string from_domain;
            std::string utfizedDomain = recognizer->Utfize(std::string(value.begin(), value.end()));
            find_domain(utfizedDomain, from_domain);
            result.parameters.from_domain = from_domain;
            if (from_domain.empty()) {
                std::string from;
                boost::range::copy(value, std::back_inserter(from));
                NWLOG(warning, gconfig->clusterName, gconfig->hostName,
                    NLog::from=from,
                    NLog::message="domain not found in header from")
            }

            if (envelope->address_sender_from_mailbody.empty()) {
                mailbox_list mailboxes;
                if (parse_address(value, mailboxes)) {
                    if (!mailboxes.empty()) {
                        envelope->address_sender_from_mailbody.assign(
                            mailboxes.front().second.begin(),
                            mailboxes.front().second.end());
                    }
                } else {
                    envelope->address_sender_from_mailbody = {};
                }
            }
        } else if (lname == "sender") {
            if (!envelope->header_sender_) {
                envelope->header_sender_ = value;
            }
        } else if (lname == "x-yandex-spam") {
            if (gconfig->msgOpts.trustHeaders.count("x-yandex-spam")) {
                result.parameters.spam_status = NSO::ResolutionFromString({value.begin(), value.end()});
            }
        } else if (lname == "x-yandex-avir") {
            // TODO: MAILDLV-2861
            if (gconfig->msgOpts.trustHeaders.count("x-yandex-avir")) {
                try {
                    result.parameters.virus_status = avir_client::status(boost::lexical_cast<std::int32_t>(value));
                } catch (...) {
                    return;
                }
            }
        } else if (lname == "x-spam-flag") {
            if (gconfig->msgOpts.trustHeaders.count("x-spam-flag")) {
                result.parameters.spam_status = FromSpamFLag({value.begin(), value.end()});
            }
        } else if (lname == "x-yandex-captcha-entered") {
            if (gconfig->msgOpts.trustHeaders.count("x-yandex-captcha-entered")) {
                result.parameters.captcha = boost::iequals(value, "yes");
            }
        } else if (lname == "x-request-id") {
            result.parameters.request_id = std::string {value.begin(), value.end()};
        } else if (lname == "x-yandex-fwd") {
            if (std::size_t counter;
                boost::conversion::try_lexical_convert<std::size_t>(boost::copy_range<std::string>(value), counter)
            ) {
                result.parameters.decycler_counter = counter;
            }
        } else if (MAILER_HEADER_NAMES.count(lname) != 0) {
            result.parameters.mailer_hdr_name = lname;
            result.parameters.mailer = std::string {value.begin(), value.end()};
        }

        // add a header field only if we don't have to remove it from the message
        if (!gconfig->msgOpts.removeHeadersSet.count(lname)) {
            result.headers.push_back({name, value, hdr.GetWhole()});
        }
    });

    return result;
}

}  // namespace NNwSmtp
