#include "error_code.h"
#include "so.h"
#include "utils.h"

#include <yplatform/encoding/base64.h>

#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/so/utils.h>

#include <boost/range/algorithm.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>

#include <format>

namespace NNwSmtp::NAsyncDlv {

namespace {

std::string MakeHeader(const std::string& header, const std::string& value) {
    return std::format("{}: {}\r\n", header, value);
}

} // namespace anonymous

NSO::TEmailInfo BuildMailFrom(const envelope_ptr& envelope) {
    NSO::TEmailInfo mailFrom;
    mailFrom.Address = NSO::TEmail {.Email = envelope->m_sender};

    if (const auto& senderInfo = envelope->senderInfo; senderInfo) {
        mailFrom.Suid = std::to_string(senderInfo->Suid);
        mailFrom.Uid = senderInfo->Uid;
        mailFrom.Country = senderInfo->Country;
        mailFrom.Karma = std::to_string(senderInfo->Karma);
        mailFrom.KarmaStatus = std::to_string(senderInfo->KarmaStatus);
    }
    return mailFrom;
}

std::vector<NSO::TEmailInfo> BuildRecipients(const envelope_ptr& envelope) {
    auto result
        = boost::make_iterator_range(envelope->m_rcpts.begin(), envelope->m_rcpts.end())
        | boost::adaptors::filtered([&envelope](const auto& rcpt) {
              return envelope->m_sender_uid.empty()
                  || !rcpt.IsAccepted()
                  || rcpt.m_uid != envelope->m_sender_uid
                  || !envelope->sender_added_for_control;
          })
        | boost::adaptors::transformed([](const auto& rcpt) {
              NSO::TEmailInfo emailInfo;
              emailInfo.Address = NSO::TEmail {.Email = rcpt.m_name};
              emailInfo.IsMaillist = rcpt.is_maillist_;

              if (!rcpt.m_uid.empty()) {
                  emailInfo.Uid = rcpt.m_uid;
              }

              if (rcpt.m_suid) {
                  emailInfo.Suid = std::to_string(rcpt.m_suid);
              }

              if (!rcpt.country_.empty()) {
                  emailInfo.Country = rcpt.country_;
              }

              return emailInfo;
          });

    return {result.begin(), result.end()};
}

NSO::TRequestPtr BuildSoRequest(
    const TRequest& dlvRequest,
    const std::string& authResultsHeaderValue,
    dmarc::record::policy dmarcPolicy,
    const std::string& dmarcDomain,
    const NSO::TOptions& options
) {
    std::optional<std::string> spfOpt;
    if (!dlvRequest.Envelope->senderInfo && dlvRequest.SpfResult && dlvRequest.SpfExpl) {
        spfOpt = std::format("{} ({}) envelope-from={}", dlvRequest.SpfResult.value(), dlvRequest.SpfExpl.value(), dlvRequest.SmtpFrom);
    }

    std::optional<std::string> dmarcOpt;
    if (dmarcPolicy != dmarc::record::UNKNOWN) {
        dmarcOpt = std::format("{} {}", dmarc::convert_policy(dmarcPolicy), dmarcDomain);
    }

    std::optional<std::string> authResultsOpt;
    if (!authResultsHeaderValue.empty()) {
        authResultsOpt = authResultsHeaderValue;
    }

    std::optional<std::string> messageIdOpt;
    if (dlvRequest.Envelope->added_message_id_) {
        messageIdOpt = dlvRequest.Envelope->message_id_;
    }

    return NSO::BuildSoHttpRequest({
        .SessionId = dlvRequest.SessionId,
        .EnvelopeId = dlvRequest.Envelope->m_id,
        .RemoteHost = dlvRequest.RemoteHost,
        .RemoteIp = dlvRequest.Ip.to_string(),
        .HeloHost = dlvRequest.HeloHost,
        .MailFrom = BuildMailFrom(dlvRequest.Envelope),
        .Rcpts = BuildRecipients(dlvRequest.Envelope),
        .Timestamp = GetUnixTimestamp(),
        .SoClusterName = options.ClusterName,
        .Spf = std::move(spfOpt),
        .Dmarc = std::move(dmarcOpt),
        .AuthResults = std::move(authResultsOpt),
        .MessageId = std::move(messageIdOpt),
        .Headers = *dlvRequest.Envelope->header_storage_,
        .InternalHeaders = options.InternalHeaders,
        .MessageBody = GetMessageBodyRange(dlvRequest.Envelope),
        .IsDryRun = options.DryRun,
    });
}

bool CheckInactivity(
    const TRcpt& rcpt,
    const std::set<std::string>& foundTypes,
    const NSO::TOptions& options
) {
    if (options.RejectForInactive && !rcpt.is_maillist_) {
        if (auto regDate = std::chrono::system_clock::from_time_t(rcpt.registrationDate);
            std::chrono::system_clock::now() < regDate + options.NewUserPeriod
        ) {
            return true;
        } else if (options.InactivityThreshold == 0 ||
            options.InactivityThreshold >= rcpt.inactive_for_
        ) {
            return true;
        } else if (rcpt.m_spam_status == NSO::EResolution::SO_RESOLUTION_REJECT ||
            rcpt.m_spam_status == NSO::EResolution::SO_RESOLUTION_SPAM
        ) {
            return false;
        }

        for (const auto& types : options.RejectedTypes) {
            if (types.empty()) {
                continue;
            }

            if (std::includes(foundTypes.begin(), foundTypes.end(), types.begin(), types.end())) {
                return false;
            }
        }
    }

    return true;
}

void CheckInactives(
    const NSO::TResponse& response,
    const envelope_ptr& envelope,
    const NSO::TOptions& options
) {
    if (!response.ActivityInfos) {
        return;
    }

    std::set<std::string> foundTypes = {
        response.SoClasses.begin(),
        response.SoClasses.end(),
    };

    boost::range::for_each(
        response.ActivityInfos.value(),
        [&](const auto& activityInfo) {
            auto* rcpt = envelope->m_rcpts.GetByUid(activityInfo.Uid);
            if (!rcpt || !activityInfo.Status) {
                return;
            }
            rcpt->inactive_for_ = activityInfo.Status.value();

            bool inactivityStatus = CheckInactivity(*rcpt, foundTypes, options);

            if (!inactivityStatus) {
                rcpt->m_delivery_status = check::CHK_REJECT;
                rcpt->m_remote_answer = "552 5.2.2 Mailbox size limit exceeded";
            }
        }
    );
}

TErrorCode GetMaliciousErrorCode(const std::vector<std::string>& soClasses) {
    auto soClass = boost::range::find_if(soClasses,
        [](const auto& soClass) {
            return boost::starts_with(soClass, BAN_REASON_TYPE);
        }
    );

    if (soClass == soClasses.end()) {
        return EError::MaliciousRejected;
    }

    if (auto reason = *soClass; reason == RFC_FAIL) {
        return EError::RfcFailRejected;
    } else if (reason == URL_RBL) {
        return EError::UrlRblRejected;
    } else if (reason == BAD_KARMA) {
        return EError::BadKarmaRejected;
    } else if (reason == MAIL_LIMITS) {
        return EError::MailLimitsRejected;
    } else if (reason == PDD_ADMIN_KARMA) {
        return EError::PddAdminKarmaRejected;
    } else if (reason == BOUNCES) {
        return EError::BouncesRejected;
    } else if (reason == SPAM_COMPL) {
        return EError::SpamComplRejected;
    } else {
        return EError::MaliciousRejected;
    }
}

std::pair<TErrorCode, std::string> MakeSoStatus(
    NSO::EResolution resolution,
    const std::vector<std::string>& soClasses,
    const std::string& dmarcDomain,
    const dmarc::record::policy& dmarcPolicy,
    const NSO::TOptions& options,
    bool captcha
) {
    std::string answer;
    TErrorCode ec;

    if (resolution == NSO::EResolution::SO_RESOLUTION_INVALID) {
        ec = EError::SoTempFail;
        answer = "451 4.7.1 Service unavailable - try again later";
    } else if (resolution == NSO::EResolution::SO_RESOLUTION_REJECT) {
        if (options.ActionMalicious == NSO::ESpamPolicy::REJECT) {
            ec = GetMaliciousErrorCode(soClasses);
            if (dmarcPolicy == dmarc::record::REJECT) {
                answer = "550 5.7.1 Email rejected per DMARC policy for " + dmarcDomain;
            } else {
                answer = "554 5.7.1 " + options.ReplyTextMalicious;
            }
        } else {
            ec = EError::MaliciousDiscarded;
            if (!options.ReplyTextDiscard.empty()) {
                answer = "250 2.0.0 " + options.ReplyTextDiscard;
            }
        }
    } else if (resolution == NSO::EResolution::SO_RESOLUTION_SPAM) {
        if (options.ActionSpam == NSO::ESpamPolicy::REJECT && !captcha) {
            ec = EError::SpamRejected;
            if (dmarcPolicy == dmarc::record::REJECT) {
                answer = "550 5.7.1 Email rejected per DMARC policy for " + dmarcDomain;
            } else {
                answer = "554 5.7.1 " + options.ReplyTextSpam;
            }
        } else if (options.ActionSpam == NSO::ESpamPolicy::DISCARD) {
            ec = EError::SpamDiscarded;
            if (!options.ReplyTextDiscard.empty()) {
                answer = "250 2.0.0 " + options.ReplyTextDiscard;
            }
        }
    }
    return std::make_pair(std::move(ec), std::move(answer));
}

void MakeSoPersonalStatus(const NSO::TResponse& response, const envelope_ptr& envelope) {
    boost::range::for_each(response.PersonalResolutions.value(), [&envelope](const auto& personal) {
        if (auto* rcpt = envelope->m_rcpts.GetByUid(personal.Uid)) {
            rcpt->m_spam_status = personal.Resolution;
            rcpt->so_classes = personal.SoClasses;

            std::string hint = "email=" + rcpt->m_name + "\nreplace_so_labels=1\n" +
                boost::join(
                    personal.SoClasses
                        | boost::adaptors::transformed([](const auto& value) { return "label=SystMetkaSO:" + value; }),
                "\n").append("\n");
            envelope->xyandex_hint_header_value_personal_.push_back(std::move(hint));
        }
    });
}

std::string MakeUidStatusHeader(const NSO::TResponse& response, const envelope_ptr& envelope) {
    auto header = boost::join(response.PersonalResolutions.value()
        | boost::adaptors::filtered([&envelope](const auto& personal) {
            return envelope->m_rcpts.GetByUid(personal.Uid) != nullptr;
        })
        | boost::adaptors::transformed([](const auto& personal) {
            return NSO::ResolutionToString(personal.Resolution) + " " + personal.Uid;
        }), ",");
    return header.empty() ? "" : MakeHeader("X-Yandex-Uid-Status", header);
}

bool IsSpam(NSO::EResolution resolution) {
    return resolution == NSO::EResolution::SO_RESOLUTION_REJECT || resolution == NSO::EResolution::SO_RESOLUTION_SPAM;
}

std::string MakePersonalSpamHeader(const NSO::TResponse& response, const envelope_ptr& envelope) {
    auto header = boost::join(response.PersonalResolutions.value()
        | boost::adaptors::filtered([&envelope](const auto& personal) {
            return (envelope->m_rcpts.GetByUid(personal.Uid) != nullptr) &&
                (boost::range::find(personal.SoClasses, "pf_spam") != personal.SoClasses.end());
        })
        | boost::adaptors::transformed([](const auto& personal) {
            return personal.Uid;
        }), "\n");
    return header.empty() ? "" : MakeHeader("X-Yandex-Personal-Spam", yplatform::base64_encode_str(header));
}

std::string MakeSpamHeader(NSO::EResolution resolution) {
    return MakeHeader("X-Yandex-Spam", NSO::ResolutionToString(resolution));
}

std::string MakeForeignMx(const NSO::TResponse& response) {
    if (auto& outParameters = response.OutParameters;
        !outParameters ||
        !outParameters->Type
    ) {
        return {};
    }
    if(auto type = response.OutParameters->Type; type == NSO::EForwardType::FORWARD_TYPE_WHITE) {
        return "whi";
    } else if(type == NSO::EForwardType::FORWARD_TYPE_MXBACK) {
        return "bla";
    } else {
        return "gre";
    }
}

std::string MakeSpamFlagHeader(NSO::EResolution resolution) {
    return MakeHeader("X-Spam-Flag", IsSpam(resolution) ? "YES": "NO");
}

std::string MakeForeignMXHeader(std::string foreignMX) {
    return MakeHeader("X-Yandex-ForeignMX", foreignMX);
}

std::string FormatSoClasses(const NSO::TResponse& response) {
    return boost::join(response.SoClasses
        | boost::adaptors::transformed([](const auto& soClass) {
            if (boost::starts_with(soClass, "domain_")) {
                return "label=" + soClass;
            } else {
                return "label=SystMetkaSO:" + soClass;
            }
        }), "\n").append("\n");
}

std::pair<TErrorCode, std::string> MakeSoResult(
    const NSO::TResponse& response,
    const envelope_ptr& envelope,
    const std::string& dmarcDomain,
    const dmarc::record::policy& dmarcPolicy,
    const NSO::TOptions& options,
    bool captcha
) {
    TErrorCode ec;
    std::string answer;
    std::tie(ec, answer) = MakeSoStatus(response.Resolution, response.SoClasses, dmarcDomain,
        dmarcPolicy, options, captcha);

    if (response.PersonalResolutions) {
        MakeSoPersonalStatus(response, envelope);
    }

    CheckInactives(response, envelope, options);

    envelope->so_status = response.Resolution;
    envelope->m_spam = IsSpam(response.Resolution);

    envelope->foreign_mx = MakeForeignMx(response);
    envelope->so_labels_ = {response.SoClasses.begin(), response.SoClasses.end()};
    envelope->xyandex_hint_header_value_ += FormatSoClasses(response);

    auto hdr = MakeSpamHeader(response.Resolution);

    if (!envelope->foreign_mx.empty()) {
        hdr += MakeForeignMXHeader(envelope->foreign_mx);
    }

    if (response.PersonalResolutions) {
        hdr += MakeUidStatusHeader(response, envelope);
        hdr += MakePersonalSpamHeader(response, envelope);
    }

    if (options.AddXspamFlag) {
        hdr += MakeSpamFlagHeader(response.Resolution);
    }

    envelope->so_headers = hdr;
    NUtil::AppendToSegment(envelope->added_headers_, hdr);
    return std::make_pair(std::move(ec), std::move(answer));
}

void LogSoStatus(const TContextPtr& context, const envelope_ptr& envelope) {
    NWLOG_CTX(notice, context, "SOCHECK", "headers=" + envelope->so_headers);

    NWLOG_CTX(notice, context, "SOCHECK", "from=" + envelope->m_sender + ", status=" +
        NSO::Report(envelope->so_status) + ", so_classes=" + boost::join(envelope->so_labels_, ",") +
        ", foreign_mx=" + envelope->foreign_mx);

    boost::for_each(
        boost::make_iterator_range(envelope->m_rcpts.begin(), envelope->m_rcpts.end())
            | boost::adaptors::filtered([](const auto& rcpt) {
                return rcpt.m_spam_status != NSO::EResolution::SO_RESOLUTION_SKIP;
            }),
        [&context](const auto& rcpt) {
            NWLOG_CTX(notice, context, "SOCHECK", "rcpt=" + rcpt.m_name + ", status=" +
                NSO::Report(rcpt.m_spam_status) + ", so_classes=" + boost::join(rcpt.so_classes, ","));
        }
    );
}

} // namespace NNwSmtp::NAsyncDlv
