#include "error_code.h"
#include "utils.h"

#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/options.h>
#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/src/xyandexhint/xyandexhint.h>

#include <mail/sova/include/nwsmtp/decoder/mimedecoder.h>

#include <yplatform/zerocopy/segment.h>

#include <boost/algorithm/string.hpp>
#include <boost/tokenizer.hpp>

#include <vector>
#include <sstream>

namespace NNwSmtp::NAsyncDlv {

SmtpPoint CheckSenderDependentRelayMap(const envelope_ptr& envelope, const std::string& smtpFrom) {
    const auto& map = gconfig->delivery.senderDependent.relaysMap;
    auto it = map.find(smtpFrom);

    if (it == map.end()) {
        std::string::size_type at = smtpFrom.find('@');
        std::string atdomain;
        if (at != std::string::npos && at != smtpFrom.size() - 1)
            atdomain.assign(smtpFrom, at + 1, std::string::npos);
        it = map.find(atdomain);
    }

    if (gconfig->msgOpts.rewriteSenderFromHeader && !envelope->address_sender_from_mailbody.empty()) {
        const auto& from = envelope->address_sender_from_mailbody;
        if (it == map.end()) {
            it = map.find(from);
        }
        if (it == map.end()) {
            std::string::size_type at = from.find('@');
            std::string atdomain;
            if (at != std::string::npos && at != from.size() - 1) {
                atdomain.assign(from, at + 1, std::string::npos);
            }
            it = map.find(atdomain);
        }
    }

    if (it != map.end()) {
        return it->second;
    }

    return {};
}

bool CanSendDsn(const envelope_ptr& envelope, std::int32_t mode) {
    return (gconfig->dsn.mode & mode) && !envelope->m_sender.empty();
}

std::pair<TErrorCode, std::string> CalculateResponse(const envelope_ptr& envelope, bool dsn) {
    std::string answer;
    TErrorCode ec;
    bool accepted = false;
    bool rejected = false;
    bool tempfailed = false;
    std::string rejectAnswer;
    std::string tempfailAnswer;
    for (const auto& rcpt : envelope->m_rcpts) {
        if (rcpt.m_delivery_status == check::CHK_ACCEPT) {
            accepted = true;
        }
        else if (rcpt.m_delivery_status == check::CHK_REJECT) {
            rejected = true;
            rejectAnswer = rcpt.m_remote_answer;
        }
        else if (rcpt.m_delivery_status == check::CHK_TEMPFAIL) {
            tempfailed = true;
            tempfailAnswer = rcpt.m_remote_answer;
        }
    }
    if (accepted && (dsn || (!rejected && !tempfailed))) {
        answer.clear();
    } else if (rejected) {
        // If one of the recipients was rejected while no one was accepted
        // we can report message rejection.
        ec = EError::DeliveryRejected;
        answer = std::move(rejectAnswer);
    } else {
        ec = EError::DeliveryTempFail;
        answer = std::move(tempfailAnswer);
    }
    return std::make_pair(std::move(ec), std::move(answer));
}

void RemoveHeaders(envelope_ptr& envelope) {
    if (gconfig->delivery.removeHeaders.empty()) {
        return;
    }
    auto [headers, body] = ParseMessage(envelope->altered_message_);
    const auto& allHeaders = headers.GetAllHeaders();

    std::string newMsg;
    newMsg.reserve(envelope->altered_message_.size());

    boost::range::for_each(allHeaders, [&](const auto& hdr) {
        std::string lname;
        lname.reserve(hdr.Name.size());
        std::transform(hdr.Name.begin(), hdr.Name.end(),
            std::back_inserter(lname), ::tolower);
        if (gconfig->delivery.removeHeaders.contains(lname)) {
            return;
        }
        newMsg += std::string{hdr.GetWhole().begin(), hdr.GetWhole().end()} + "\r\n";
    });
    newMsg += "\r\n";
    newMsg += std::string{envelope->orig_message_body_beg_, envelope->orig_message_.cend()};
    envelope->altered_message_ = std::move(newMsg);
}

std::pair<TErrorCode, std::string> ProcessAvirStatus(avir_client::status status) {
    std::string answer;
    TErrorCode ec;
    if (status == avir_client::infected) {
        if (gconfig->avir.action_virus == 0) {
            ec =  EError::AvirDiscarded;
            if (!gconfig->avir.reply_text_discard.empty()) {
                answer = "250 2.0.0 " + gconfig->avir.reply_text_discard;
            }
        } else {
            ec = EError::AvirRejected;
            answer = "554 5.7.1 " + gconfig->avir.reply_text;
        }
    }

    return std::make_pair(std::move(ec), std::move(answer));
}

TBufferRange GetMessageBodyRange(const envelope_ptr& envelope) {
    return boost::make_iterator_range(
        envelope->orig_message_body_beg_,
        envelope->orig_message_.cend()
    );
}

std::string GetMessageBody(const TBufferRange& messageBufferRange) {
    return {
        messageBufferRange.begin(),
        messageBufferRange.end(),
    };
}

} // namespace NwSmtp::NAsyncDlv
