#include "errors.h"

#include <mail/nwsmtp/src/nsls/error_code.h>
#include <mail/nwsmtp/src/so/errors.h>

#include <mail/ymod_mds/src/errors.h>

namespace NNwSmtp::NDlv {

const char* TCategory::name() const noexcept {
    return "NNwSmtp::NDlv::TCategory";
}

std::string TCategory::message(int value) const {
    switch (static_cast<EError>(value)) {
        case EError::Ok:
            return "ok";
        case EError::BadMessage:
            return "bad message";
        case EError::NslsPermanentError:
            return "nsls permanent error";
        case EError::NslsTemporaryError:
            return "nsls temporary error";
        case EError::DuplicateFound:
            return "duplicate found";
        case EError::CycleDetected:
            return "loop detected";
        case EError::Exception:
            return "exception";
        case EError::MdsError:
            return "mds temporary error";
        case EError::StorageMailNotFound:
            return "message not found in the storage";
        case EError::SOError:
            return "so temporary error";
        case EError::AvirError:
            return "avir temporary error";
        case EError::Virus:
            return "virus message";
        case EError::Spam:
            return "spam message";
        case EError::Malicious:
            return "malicious spam message";
        case EError::InvalidFid:
            return "nsls permanent error";
    }

    return "unknown error (" + std::to_string(value) + ")";
}

bool TCategory::equivalent(const boost::system::error_code& code, int condition) const noexcept {
    switch (EError(condition)) {
        case EError::MdsError:
            return code == NMds::EError::StorageError;
        case EError::StorageMailNotFound:
            return code == NMds::EError::StorageMailNotFound;
        case EError::NslsPermanentError:
            return code == NNsls::EError::EC_PERM_ERROR;
        case EError::NslsTemporaryError:
            return code == NNsls::EError::EC_PARSE_ERROR
                || code == NNsls::EError::EC_BAD_STATUS;
        case EError::SOError:
            return code == NSO::EError::ParseError
                || code == NSO::EError::BadStatus;
        default:
            return boost::system::error_category::equivalent(code, condition);
    }
}

const boost::system::error_category& category() noexcept {
    static TCategory instance;
    return instance;
}

boost::system::error_code make_error_code(EError error) {
    return {static_cast<int>(error), category()};
}

boost::system::error_condition make_error_condition(EError ec) {
    return {static_cast<int>(ec), category()};
}

}  // namespace NNwSmtp::NDlv
