#include "hint.h"

#include <boost/algorithm/string.hpp>


namespace NNwSmtp::NDlv {

using NHint::TParamsContainer;
using NHint::ParseBase64;

const std::string SYSTEM_PREFIX = "SystMetkaSO:";
const std::string DOMAIN_PREFIX = "domain_";
const std::string SYMBOL_PREFIX = "symbol:";

void WriteToHint(const TLabels& labels, TParamsContainer& params) {
    for (const auto& lid : labels.Lids.value_or(std::vector<std::string>{})) {
        params.insert({"lid", lid});
    }
    for (auto label : labels.System.value_or(std::vector<std::string>{})) {
        if (!boost::starts_with(label, SYSTEM_PREFIX)) {
            label = SYSTEM_PREFIX + label;
        }
        params.insert({"label", label});
    }
    for (const auto& label : labels.Imap.value_or(std::vector<std::string>{})) {
        params.insert({"imaplabel", label});
    }
    for (auto label : labels.Symbol.value_or(std::vector<std::string>{})) {
        if (!boost::starts_with(label, SYMBOL_PREFIX)) {
            label = SYMBOL_PREFIX + label;
        }
        params.insert({"label", label});
    }
    for (const auto& label : labels.User.value_or(std::vector<std::string>{})) {
        params.insert({"userlabel", label});
    }
}

void WriteToHint(const TMailInfo& mailInfo, TParamsContainer& params) {
    if (!mailInfo.Fid.empty()) {
        params.insert({"fid", mailInfo.Fid});
    } else {
        if (mailInfo.FolderPath && !mailInfo.FolderPath->empty()) {
            params.insert({"folder_path", mailInfo.FolderPath.value()});
        }
        if (mailInfo.FolderSpamPath && !mailInfo.FolderSpamPath->empty()) {
            params.insert({"folder_spam_path", mailInfo.FolderSpamPath.value()});
        }
        if (mailInfo.FolderPathDelim && !mailInfo.FolderPathDelim->empty()) {
            params.insert({"folder_path_delim", mailInfo.FolderPathDelim.value()});
        }
    }

    if (mailInfo.ReceivedDate) {
        params.insert({"received_date", std::to_string(mailInfo.ReceivedDate.value())});
    }

    WriteToHint(mailInfo.Labels.value_or(TLabels{}), params);

    if (!mailInfo.ExternalImapId.empty()) {
        params.insert({"external_imap_id", mailInfo.ExternalImapId});
    }
    if (mailInfo.OldMid && !mailInfo.OldMid->empty()) {
        params.insert({"mid", mailInfo.OldMid.value()});
    }
}

void WriteToHint(const TOptions& options, TParamsContainer& params) {
    params.insert({"skip_loop_prevention", "1"});
    params.insert({"filters", std::to_string(options.UseFilters.value_or(false))});
    params.insert({"disable_push", std::to_string(!options.EnablePush.value_or(false))});
    params.insert({"allow_duplicates", std::to_string(options.AllowDuplicates.value_or(false))});
    params.insert({"imap", std::to_string(options.DlvType == EDlvType::Imap)});
    params.insert({"skip_meta_msg", std::to_string(options.SkipMetaMsg)});
}

TParamsContainer BuildHint(
    const TOptions& options,
    const TMailInfo& mailInfo,
    const std::optional<NSO::TResponse>& soResult
) {
    TParamsContainer params;

    WriteToHint(options, params);
    WriteToHint(mailInfo, params);

    params.insert({"sync_dlv", std::to_string(true)});
    // TODO remove after MAILDLV-4960
    params.insert({"no_such_fid_fail", std::to_string(true)});

    if (soResult) {
        for (auto label : soResult->SoClasses) {
            label = boost::starts_with(label, DOMAIN_PREFIX) ? label : (SYSTEM_PREFIX + label);
            params.insert({"label", label});
        }
    }

    return params;
}

}  // namespace NNwSmtp::NDlv
