#include "errors.h"
#include "hint.h"
#include "sync.h"
#include "utils.h"

#include <mail/yplatform/include/yplatform/application/find.h>
#include <mail/ymod_httpclient/include/ymod_httpclient/cluster_client.h>

#include <mail/nwsmtp/src/avir/client_impl.h>
#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/nsls/client_impl.h>
#include <mail/nwsmtp/src/mds/client_impl.h>
#include <mail/nwsmtp/src/resolver/client_impl.h>
#include <mail/nwsmtp/src/resolver/resolver.h>
#include <mail/nwsmtp/src/so/client_impl.h>
#include <mail/nwsmtp/src/utils.h>

#include <mail/ymod_mds/src/client.h>

#include <yplatform/yield.h>

namespace NNwSmtp::NDlv {

static const std::string SYNC_DLV{"SYNC-DLV"};

TSyncDelivery::TSyncDelivery(
    TContextPtr context,
    const TOptions& options,
    const TEnvelopeInfo& envInfo,
    const TMailInfo& mailInfo,
    const TUserInfo& userInfo,
    TBuffer message,
    NNsls::TNslsClientPtr nslsClient,
    NMds::TMdsClientPtr mdsClient,
    NSO::TSOClientPtr soInClient,
    NSO::TSOClientPtr soOutClient,
    NDns::TResolverClientPtr resolverClient,
    NAvir::TAvirCheckClientPtr avirCheckClient,
    TConfigPtr config,
    TSyncResultCallback&& callback
)
    : Context(std::move(context))
    , Options(options)
    , EnvInfo(envInfo)
    , MailInfo(mailInfo)
    , UserInfo(userInfo)
    , Message(message)
    , NslsClient(std::move(nslsClient))
    , MdsClient(std::move(mdsClient))
    , SOInClient(std::move(soInClient))
    , SOOutClient(std::move(soOutClient))
    , ResolverClient(std::move(resolverClient))
    , AvirCheckClient(std::move(avirCheckClient))
    , Config(config)
    , Callback(std::move(callback))
{
}

void TSyncDelivery::operator()(TYieldCtx yieldCtx) {
    try {
        reenter(yieldCtx) {
            NWLOG_L(notice, SYNC_DLV, "userInfo: " + ToString(UserInfo));
            NWLOG_L(notice, SYNC_DLV, "mailInfo: " + ToString(MailInfo));
            NWLOG_L(notice, SYNC_DLV, "options: " + ToString(Options));

            if (Options.DlvType == EDlvType::Restore) {
                NWLOG_L(notice, SYNC_DLV, "restore message");
            } else {
                NWLOG_L(notice, SYNC_DLV, "store message: size=" + std::to_string(Message.size()));
            }

            yield ResolveIp(yieldCtx.capture(ErrorCode, EnvInfo.RemoteHost));
            if (ErrorCode) {
                NWLOG_L(error, SYNC_DLV, "error on resolve remote host, fallback to ip");
                EnvInfo.RemoteHost = EnvInfo.RemoteIp;
            }

            if (Options.DlvType == EDlvType::Restore) {
                yield GetMessage(yieldCtx.capture(ErrorCode, RawMessage));
                if (ErrorCode) {
                    NWLOG_L(error, SYNC_DLV, "mds get ended with error");
                    yield break;
                }

                Message = NUtil::MakeSegment(RawMessage);
                std::tie(Headers, Body) = ParseMessage(Message);
                Stid = MailInfo.Stid;

                NWLOG_L(notice, SYNC_DLV, "mds get successful, stid=" + Stid);

                yield CheckSpam(yieldCtx.capture(ErrorCode, SoResponse));
                if (ErrorCode) {
                    NWLOG_L(error, SYNC_DLV, "so check ended with error");
                    yield break;
                }

                ChangeTab(SoResponse, MailInfo.Tab);
            } else {
                std::tie(OriginalHeaders, Body) = ParseMessage(Message);
                if (OriginalHeaders.Empty()) {
                    NWLOG_L(error, SYNC_DLV, "parse message: no headers");
                    ErrorCode = EError::BadMessage;
                    yield break;
                }

                PrepareHeaders();

                if (DetectCycle(Config, Options, DecyclerCounter)) {
                    NWLOG_L(error, SYNC_DLV, "too many hops, count=" + std::to_string(DecyclerCounter));
                    ErrorCode = EError::CycleDetected;
                    yield break;
                }

                if (Options.DetectSpam && Options.DetectSpam.value()) {
                    yield CheckSpam(yieldCtx.capture(ErrorCode, SoResponse));
                    if (ErrorCode) {
                        NWLOG_L(error, SYNC_DLV, "so check ended with error");
                        yield break;
                    }

                    NWLOG_L(notice, "SOCHECK", "status=" + SoResultToString(SoResponse));

                    ErrorCode = MakeSoResult(
                        SoResponse,
                        Options,
                        CaptchaWasEntered
                    );
                    if (ErrorCode) {
                        yield break;
                    }
                }

                if (Options.DetectVirus && Options.DetectVirus.value()) {
                    yield CheckVirus(yieldCtx.capture(ErrorCode, AvirResult));
                    if (ErrorCode) {
                        NWLOG_L(error, SYNC_DLV, "avir check ended with error");
                        ErrorCode = EError::AvirError;
                        yield break;
                    }

                    NWLOG_L(notice, SYNC_DLV, "avir check ended with status=" +
                        AvirResultToString(AvirResult));

                    ErrorCode = MakeAvirResult(AvirResult);
                    if (ErrorCode) {
                        yield break;
                    }
                }

                AddedHeaders = NUtil::MakeSegment(
                    BuildReceivedHeader(EnvInfo, UserInfo, Config->SoOptions.ClusterName),
                    BuildDecyclerHeader(DecyclerCounter + 1)
                );

                yield PutMessage(yieldCtx.capture(ErrorCode, Stid));
                if (ErrorCode) {
                    NWLOG_L(error, SYNC_DLV, "mds put ended with error");
                    yield break;
                }

                NWLOG_L(notice, SYNC_DLV, "mds put successful, stid=" + Stid);
            }

            yield StoreMessage(yieldCtx.capture(ErrorCode, NslsResponse));
            if (ErrorCode) {
                NWLOG_L(error, SYNC_DLV, "store message ended with error");
                if (Options.DlvType != EDlvType::Restore) {
                    NWLOG_L(notice, SYNC_DLV, "should delete stid = " + Stid);
                }
                yield break;
            }

            std::tie(ErrorCode, Result) = MakeStoreResult(NslsResponse);
            if (ErrorCode) {
                if (Options.DlvType != EDlvType::Restore) {
                    NWLOG_L(notice, SYNC_DLV, "should delete stid = " + Stid);
                }
                yield break;
            }
            NWLOG_L(notice, "DECYCLER", "x-yandex-fwd=" + std::to_string(DecyclerCounter));
            NWLOG_L(notice, SYNC_DLV, "notsolitesrv store successful, mid=" + Result.Mid);
        }
    } catch (const std::exception& e) {
        NWLOG_L_EXC(error, SYNC_DLV, "exception on sync delivery", e);
        return Callback(EError::Exception, {});
    }

    if (yieldCtx.is_complete()) {
        if (ErrorCode) {
            NWLOG_L_EC(error, SYNC_DLV, "error occurred", ErrorCode);
        }
        Callback(ErrorCode, Result);
    }
}

#include <yplatform/unyield.h>

void TSyncDelivery::PrepareHeaders() {
    if (Config->TrustHeaders.count("x-yandex-captcha-entered")) {
        CaptchaWasEntered = GetCaptchaWasEntered(OriginalHeaders);
    }
    if (Config->Decycler.updateHeader) {
        DecyclerCounter = GetDecyclerCounter(OriginalHeaders);
    }

    boost::range::for_each(OriginalHeaders.GetAllHeaders(), [this](const auto& value){
        if (boost::iequals(value.Name, "x-yandex-fwd")) {
            return;
        }
        Headers.Add(value.Name, value.Value);
    });
}

void TSyncDelivery::PutMessage(auto handler) {
    auto putRequest = NMds::TPutRequest{
        UserInfo.Uid,
        IsSpam,
        Headers,
        AddedHeaders,
        Body,
        Config->RemoveHeaders,
    };

    MdsClient->Put(Context, std::move(putRequest), std::move(handler));
}

void TSyncDelivery::GetMessage(auto handler) {
    MdsClient->Get(Context, MailInfo.Stid, std::move(handler));
}

void TSyncDelivery::CheckSpam(auto handler) {
    auto request = BuildSoRequest(
        Options,
        EnvInfo,
        UserInfo,
        MailInfo,
        Body,
        Headers,
        Config
    );

    if (Options.SoOut) {
        SOOutClient->Check(Context, std::move(request), std::move(handler));
    } else {
        SOInClient->Check(Context, std::move(request), std::move(handler));
    }
}

void TSyncDelivery::ResolveIp(auto handler) {
    ResolverClient->ResolveIp(Context, EnvInfo.RemoteIp, std::move(handler));
}

void TSyncDelivery::StoreMessage(auto handler) {
    auto hint = BuildHint(Options, MailInfo, SoResponse);
    auto request = BuildStoreRequest(Options, EnvInfo, UserInfo, Stid, IsSpam, std::move(hint));

    NslsClient->Store(Context, std::move(request), std::move(handler));
}

void TSyncDelivery::CheckVirus(auto handler) {
    AvirCheckClient->Check(Context, Message, std::move(handler));
}

void RunSyncDelivery(
    TContextPtr context,
    const TEnvelopeInfo& envInfo,
    const TMailInfo& mailInfo,
    const TUserInfo& userInfo,
    const TOptions& options,
    TBuffer message,
    TSyncResultCallback&& callback
) {
    auto& io = *yplatform::find_reactor("global")->io();

    auto nslsClient = std::make_shared<NNsls::TNslsClient>(
        yplatform::find<yhttp::cluster_client, std::shared_ptr>("nsls_client"), io);

    auto mdsModule = yplatform::find<::NMds::TClient, std::shared_ptr>("mds_client");
    mdsModule->logger(glog->yplatform);

    auto mdsClient = std::make_shared<NMds::TMdsClient>(std::move(mdsModule), io);

    auto soInModule = yplatform::find<yhttp::cluster_client, std::shared_ptr>("so_in_client");

    auto soOutModule = yplatform::find<yhttp::cluster_client, std::shared_ptr>("so_out_client");

    auto soInClient = std::make_shared<NSO::TSOClient>(std::move(soInModule), NSO::ESOType::SOIn, io);

    auto soOutClient = std::make_shared<NSO::TSOClient>(std::move(soOutModule), NSO::ESOType::SOOut, io);

    auto resolverModule = yplatform::find<NDns::TResolver, std::shared_ptr>("resolver");
    resolverModule->logger(glog->yplatform);

    auto resolverClient = std::make_shared<NDns::TResolverClient>(std::move(resolverModule), io);

    auto avirCheckClient = std::make_shared<NAvir::TAvirCheckClient>(
        std::make_shared<avir_client>(io, gconfig->avir.client_opts), io);

    auto config = MakeConfig(gconfig);

    auto sync = std::make_shared<TSyncDelivery>(std::move(context), options, envInfo, mailInfo, userInfo,
        message, std::move(nslsClient), std::move(mdsClient), std::move(soInClient), std::move(soOutClient),
        std::move(resolverClient), std::move(avirCheckClient), std::move(config), std::move(callback));
    yplatform::spawn(io, sync);
}

}  // namespace NNwSmtp::NDlv
