#include "errors.h"
#include "hint.h"
#include "utils.h"

#include <mail/library/received_header/received_header.h>

#include <mail/nwsmtp/src/so/utils.h>

namespace NNwSmtp::NDlv {

NNsls::TRequest BuildStoreRequest(
    const TOptions& options,
    const TEnvelopeInfo& envInfo,
    const TUserInfo& userInfo,
    const std::string& stid,
    bool isSpam,
    NHint::TParamsContainer hint
) {
    NNsls::TRequest request {};
    request.envelope.session_id = envInfo.SessionId;
    request.envelope.envelope_id = envInfo.EnvId;
    request.envelope.mail_from.email = "<>";

    request.envelope.remote_ip = envInfo.RemoteIp;
    request.envelope.remote_host = envInfo.RemoteHost;
    request.envelope.helo = envInfo.RemoteHost;

    request.message.spam = isSpam;
    request.message.stid = stid;
    request.message.timemark = NUtil::GetNowMs();

    request.recipients.push_back(NNsls::TRecipient {
        userInfo.Email,
        userInfo.Uid,
        NNsls::ELocal::yes,
        NNsls::TNotification {false, false, false},
        options.DlvType == EDlvType::Mailish,
        std::nullopt,
        std::nullopt
    });

    NNsls::THint nslsHint;
    boost::range::for_each(hint, [&](auto& value) {
        nslsHint[value.first].emplace_back(value.second);
    });

    request.message.hints.emplace_back(std::move(nslsHint));

    request.requestId = envInfo.RequestId;

    return request;
}

std::string BuildReceivedHeader(
    const TEnvelopeInfo& envInfo,
    const TUserInfo& userInfo,
    const std::string& clusterName
) {
    NReceived::TReceived received;
    received.Date = NUtil::GetRfc822DateNow();

    received.RemoteIp = envInfo.RemoteIp;
    received.RemoteHost = envInfo.RemoteHost;
    received.HeloHost = envInfo.RemoteHost;

    received.LocalHost = boost::asio::ip::host_name();
    received.Protocol = "HTTP";
    received.SessionId = envInfo.SessionId + "-" + envInfo.EnvId;
    received.ClusterName = clusterName;
    received.Recipient = userInfo.Email;
    return NReceived::BuildReceivedHeader(received);
}

NSO::TRequestPtr BuildSoRequest(
    const TOptions& options,
    const TEnvelopeInfo& envInfo,
    const TUserInfo& userInfo,
    const TMailInfo& mailInfo,
    const TBufferRange& body,
    const THeaderStorageImpl& headers,
    const TConfigPtr& config
) {
    std::string timestamp;
    if (mailInfo.ReceivedDate && mailInfo.ReceivedDate.value() > 0) {
        timestamp = std::to_string(mailInfo.ReceivedDate.value());
    } else {
        timestamp = GetUnixTimestamp();
    }

    std::optional<std::string> mailish;
    if (options.DlvType == EDlvType::Mailish) {
        mailish = userInfo.Uid;
    }

    return NSO::BuildSoHttpRequest({
        .SessionId = envInfo.SessionId,
        .EnvelopeId = envInfo.EnvId,
        .RemoteHost = envInfo.RemoteHost,
        .RemoteIp = envInfo.RemoteIp,
        .EmailType = GetSoEmailType(options.DlvType),
        .Rcpts = {{
            .Address = {.Email = userInfo.Email},
            .Uid = userInfo.Uid,
        }},
        .Timestamp = std::move(timestamp),
        .SoClusterName = config->SoOptions.ClusterName,
        .Mailish = std::move(mailish),
        .Headers = headers,
        .InternalHeaders = config->SoOptions.InternalHeaders,
        .MessageBody = body,
        .IsDryRun = GetSoDryRun(options.DlvType, config->SoOptions.DryRun),
    });
}

void ChangeTab(
    std::optional<NSO::TResponse>& response,
    const std::string& tab
) {
    if (!response || tab.empty()) {
        return;
    }
    auto& classes = response->SoClasses;
    auto tabIt = std::remove_if(classes.begin(), classes.end(),
        [](const auto& label) {
            return boost::starts_with(label, "t_");
        }
    );
    auto newTab = boost::starts_with(tab, "t_") ? tab : "t_" + tab;
    if (tabIt != classes.end()) {
        classes.erase(tabIt);
    }
    classes.push_back(std::move(newTab));
}

std::pair<TErrorCode, TSyncResult> MakeStoreResult(const NNsls::TResponse& nslsResponse) {
    TSyncResult result;
    TErrorCode errorCode;
    if (nslsResponse.Status == "success") {
        if (nslsResponse.Success.IsDuplicate) {
            result.Mid = nslsResponse.Success.Mid;
            errorCode = EError::DuplicateFound;
        } else if (nslsResponse.Success.IsDecycled) {
            errorCode = EError::CycleDetected;
        } else {
            result.Mid = nslsResponse.Success.Mid;
            result.ImapId = nslsResponse.Success.ImapId;
        }
    } else if (nslsResponse.Error.IsPermanent) {
        errorCode = EError::InvalidFid;
    } else {
        errorCode = EError::NslsTemporaryError;
    }
    return {errorCode, result};
}

TErrorCode MakeAvirResult(const NAvir::TStatus& result) {
    return result == NAvir::TStatus::infected ? EError::Virus : TErrorCode{};
}

TErrorCode MakeSoResult(
    const std::optional<NSO::TResponse>& result,
    const TOptions& options,
    bool captchaWasEntered
) {
    if (!result || options.DlvType == EDlvType::Mailish) {
        return {};
    } else if (result->Resolution == NSO::EResolution::SO_RESOLUTION_SPAM) {
        if (options.DlvType == EDlvType::Save && captchaWasEntered) {
            return {};
        }
        return EError::Spam;
    } else if (result->Resolution == NSO::EResolution::SO_RESOLUTION_REJECT) {
        return EError::Malicious;
    } else {
        return {};
    }
}

std::string SoResultToString(const std::optional<NSO::TResponse>& response) {
    if (!response) {
        return {};
    } else {
        return NSO::Report(response->Resolution);
    }
}

std::string AvirResultToString(const NAvir::TStatus& result) {
    return avir_client::status_to_str(result);
}

namespace {

std::string MakeHeader(const std::string& header, const std::string& value) {
    return header + ": " + value + "\r\n";
}

} // namespace anonymous

std::string BuildDecyclerHeader(std::size_t decyclerCount) {
    return decyclerCount > 0 ? MakeHeader("X-Yandex-Fwd", std::to_string(decyclerCount)) : "";
}

std::string GetSoEmailType(EDlvType dlvType) {
    switch (dlvType) {
        case EDlvType::Mailish:
            return "EMAIL_TYPE_MAILISH";
        case EDlvType::Restore:
            return "EMAIL_TYPE_RESTORE";
        case EDlvType::Save:
            return "EMAIL_TYPE_DELAYED";
        default:
            return "EMAIL_TYPE_REGULAR";
    }
}

bool GetSoDryRun(EDlvType dlvType, bool dryRun) {
    if (dlvType == EDlvType::Mailish || dlvType == EDlvType::Restore || dryRun) {
        return true;
    }
    return false;
}

bool GetCaptchaWasEntered(const THeaderStorage<TBufferRange>& headers) {
    if (
        auto headerRange = headers.GetHeaders("x-yandex-captcha-entered");
        headerRange.empty()
    ) {
        return false;
    } else {
        return boost::iequals(headerRange.front().Value, "yes");
    }
}

std::size_t GetDecyclerCounter(const THeaderStorage<TBufferRange>& headers) {
    auto headerRange = headers.GetHeaders("x-yandex-fwd");
    if (headerRange.empty()) {
        return 0;
    }
    std::size_t counter;
    if (const std::string& value = NUtil::ToString(headerRange.front().Value);
        boost::conversion::try_lexical_convert<std::size_t>(value, counter)
    ) {
        return counter;
    }
    return 0;
}

bool DetectCycle(const TConfigPtr& config, const TOptions& options, std::size_t decyclerCounter) {
    return decyclerCounter >= config->Decycler.ttl &&
        config->Decycler.reject &&
        options.DetectCycle &&
        options.DetectCycle.value();
}

} // namespace NNwSmtp::NDlv
