#include "adkim.h"

#include <mail/nwsmtp/src/fouras/client.h>
#include <mail/nwsmtp/src/fouras/errors.h>
#include <mail/nwsmtp/src/options.h>
#include "signature_parser.h"
#include "types.h"
#include "yplatform/zerocopy/segment.h"

#include <yplatform/coroutine.h>

#define _Bool bool
#include <contrib/libs/libopendkim/libopendkim/dkim.h>
#undef _Bool

#include <boost/asio/steady_timer.hpp>
#include <boost/thread.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/predicate.hpp>

#include <nwsmtp/resolver.h>

#include <algorithm>
#include <array>
#include <deque>
#include <limits>
#include <unordered_map>
#include <unordered_set>
#include <vector>

namespace NNwSmtp {

typedef unsigned char uchar_t;

namespace {

extern "C" typedef DKIM_CBSTAT (key_lookup_func_t)(DKIM*, DKIM_SIGINFO*,
        uchar_t*, size_t);
extern "C" DKIM_CBSTAT y_dkim_final_verify(DKIM *dkim, DKIM_SIGINFO **sigs, int nsigs);
extern "C" key_lookup_func_t y_dkim_key_lookup_seed;
extern "C" key_lookup_func_t y_dkim_key_lookup_nope;

struct dkim_lib_loader : private boost::noncopyable
{
    DKIM_LIB* lib;
    dkim_lib_loader()
            : lib(dkim_init(NULL, NULL))
    {}

    ~dkim_lib_loader()
    {
        if (lib)
            dkim_close(lib);
    }
};

using std::placeholders::_1;
using std::placeholders::_2;
using std::placeholders::_3;

template <class KeyLookup>
class dkim_lib_singleton
{
    static boost::scoped_ptr<dkim_lib_loader> ptr_;
    static boost::once_flag flag_;

  public:
    static DKIM_LIB* instance() { return ptr_->lib; }

    static void init()
    {
        ptr_.reset(new dkim_lib_loader);
        dkim_set_key_lookup(ptr_->lib, KeyLookup()());

        static uint64_t clock_drift = 14400; // 4 hours
        dkim_options(ptr_->lib, DKIM_OP_SETOPT, DKIM_OPTS_CLOCKDRIFT,
            &clock_drift, sizeof(clock_drift));

        unsigned int flags = DKIM_LIBFLAGS_DELAYSIGPROC;
        dkim_options(ptr_->lib, DKIM_OP_SETOPT, DKIM_OPTS_FLAGS,
            &flags, sizeof(flags));

        dkim_set_final(ptr_->lib, y_dkim_final_verify);
    }
};

template <class T>
boost::scoped_ptr<dkim_lib_loader> dkim_lib_singleton<T>::ptr_;

template <class T>
boost::once_flag dkim_lib_singleton<T>::flag_ = BOOST_ONCE_INIT;

struct key_seed_adaptor
{
    key_lookup_func_t* operator()()
    {
        return y_dkim_key_lookup_seed;
    }
};

struct key_nope_adaptor
{
    key_lookup_func_t* operator()()
    {
        return y_dkim_key_lookup_nope;
    }
};

dkim_lib_singleton<key_seed_adaptor> lib1;
dkim_lib_singleton<key_nope_adaptor> lib2;
} // namespace

void init_dkim() {
    lib1.init();
    lib2.init();
}

namespace {

void throw_if_error(DKIM_STAT st) {
    if (st != DKIM_STAT_OK) {
        throw std::runtime_error(dkim_getresultstr(st));
    }
}

unsigned char* get_uchar_ptr(const char* orig) {
    return const_cast<unsigned char*>(reinterpret_cast<const unsigned char*>(orig));
}

void add_header(DKIM* dkim, const TBufferIterator& begin, const TBufferIterator& end) {
    size_t size = std::distance(begin, end);
    if (size == 0) {
        throw_if_error(DKIM_STAT_SYNTAX);
    }
    std::vector<char> buf;
    buf.resize(size);
    std::copy(begin, end, buf.begin());
    auto header_ptr = buf.data();

    throw_if_error(dkim_header(dkim, get_uchar_ptr(header_ptr), size));
}

void ProcessContigiousChunk(DKIM* dkim, const char* begin, const char* end) {
    size_t diff = end - begin;
    if (diff > 0) {
        throw_if_error(dkim_body(dkim, get_uchar_ptr(begin), diff));
    }
}

void set_body(DKIM* dkim, const TBufferIterator& begin, const TBufferIterator& end) {
    auto segm = yplatform::zerocopy::segment::get_part(begin, end);

    auto chunksCount = segm.end_fragment() - segm.begin_fragment();
    if (chunksCount == 0) {
        return;
    } else if (chunksCount == 1) {
        auto it = segm.head();
        auto firstFragmentEnd = segm.tail();
        ProcessContigiousChunk(dkim, it, firstFragmentEnd);
    } else {
        // first chunk as a special case
        {
            auto it = segm.head();
            auto firstFragmentEnd = (*segm.begin_fragment())->end();
            ProcessContigiousChunk(dkim, it, firstFragmentEnd);
        }

        // inner chunks
        if (chunksCount > 2) {
            auto chunkIt = std::next(segm.begin_fragment());
            auto endIt = std::prev(segm.end_fragment());
            for (; chunkIt != endIt; ++chunkIt) {
                auto it = (*chunkIt)->begin();
                auto end = (*chunkIt)->end();
                ProcessContigiousChunk(dkim, it, end);
            }
        }

        // last chunk as a special case
        {
            auto lastFragmentIt = std::prev(segm.end_fragment());
            auto it = (*lastFragmentIt)->begin();
            auto lastFragmentEnd = segm.tail();
            ProcessContigiousChunk(dkim, it, lastFragmentEnd);
        }
    }
}

bool sig_process(DKIM *dkim, DKIM_SIGINFO *sig) {
    return (dkim_sig_process(dkim, sig) == DKIM_STAT_OK) &&
        ((dkim_sig_getflags(sig) & DKIM_SIGFLAG_PASSED) != 0 || dkim_sig_geterror(sig) == DKIM_SIGERROR_BADSIG);
}

}   // namespace

struct dkim_check::impl
        : public boost::enable_shared_from_this<dkim_check::impl>,
          private boost::noncopyable
{
    typedef std::array<char, DKIM_MAXHOSTNAMELEN + 1> req_t;
    typedef std::string res_t;
    typedef std::pair<req_t, res_t> query_t;

    typedef boost::shared_ptr<dkim_check::impl> ptr_t;

    struct signature {
        dkim_check::TBufferConstIteratorRange header;
        std::vector<std::string> headerList;
        std::string selector;
        unsigned int orderNum = 0;
        query_t dnsQuery;
    };

    boost::asio::io_service::strand strand;
    resolver_t resolver;
    boost::asio::steady_timer timer;

    input p_;
    bool done_ = false;
    std::unordered_map<std::string, signature> signatures_;
    std::string primary_signature_domain_;

    impl(boost::asio::io_service& ios, const input& pp, const resolver_options& resolver_options)
        : strand(ios)
        , resolver(ios, resolver_options)
        , timer(ios)
        , p_(pp)
    {}

    void start(
        dkim_check::Handler handler,
        std::chrono::milliseconds timeout = std::chrono::milliseconds::max()
    );
    void stop();
    void cont();

private:
    void verify();
    void complete(const Output& result);

    void findSignatures();
    void addHeadersForCheck(DKIM* dkim);

    dkim_check::Handler handler;

    void startTimer(std::chrono::milliseconds timeout);
    void handleTimer(boost::system::error_code ec);
};

template <typename Handler, typename... Args>
inline void swap_and_call(Handler& handler, Args&& ...args) {
    Handler callback;
    handler.swap(callback);
    callback(std::forward<Args>(args)...);
}

namespace {
extern "C" DKIM_CBSTAT y_dkim_key_lookup_seed (DKIM *dkim, DKIM_SIGINFO *,
        uchar_t *buf, size_t buflen)
{
    typedef dkim_check::impl impl_t;

    void* ctx = dkim_get_user_context(dkim);
    if (!ctx) {
        return DKIM_STAT_NORESOURCE;
    }

    const impl_t::query_t* dnsQuery = reinterpret_cast<impl_t::query_t*>(ctx);
    const auto& res = dnsQuery->second;

    if (res.empty()) {
        return DKIM_STAT_NOKEY;
    }

    memcpy(buf, res.data(), std::min(res.size(), buflen));

    return DKIM_STAT_OK;
}

template <typename Iterator>
void y_dkim_key_lookup_collect_helper(
    const boost::system::error_code& ec,
    resolver_t::iterator_txt it,
    dkim_check::impl::ptr_t impl,
    Iterator iter)
{
    if (ec == boost::asio::error::operation_aborted || impl->done_) {
        return;
    }

    if (impl->p_.logger) {
        if (!ec) {
            impl->p_.logger(iter->second.dnsQuery.first.data() + (": " + *it));
        } else {
            impl->p_.logger(iter->second.dnsQuery.first.data() + (": " + ec.message()));
        }
    }

    if (it != resolver_t::iterator_txt()) {
        iter->second.dnsQuery.second = *it;
        ++iter;
    } else {
        if (impl->p_.logger) {
            impl->p_.logger("signature for domain " + iter->first + " skipped");
        }
        iter = impl->signatures_.erase(iter);
    }

    if (iter != impl->signatures_.end()) {
        impl->resolver.async_resolve_txt(
            iter->second.dnsQuery.first.data(),
            impl->strand.wrap([iter, impl](auto ec, auto resolveIter) {
                y_dkim_key_lookup_collect_helper(ec, resolveIter, impl, iter);
            })
        );
    } else {
        impl->cont();
    }
}

extern "C" DKIM_CBSTAT y_dkim_key_lookup_nope(DKIM* /*dkim*/,
    DKIM_SIGINFO* /*sig*/, uchar_t* /*buf*/, size_t /*buflen*/)
{
    return DKIM_STAT_NORESOURCE;
}

extern "C" DKIM_CBSTAT y_dkim_final_verify(DKIM *dkim, DKIM_SIGINFO **sigs, int nsigs) {
    typedef dkim_check::impl impl_t;

    void* ctx = dkim_get_user_context(dkim);
    if (!ctx) {
        return DKIM_CBSTAT_ERROR;
    }

    impl_t::ptr_t impl = reinterpret_cast<impl_t*>(ctx)->shared_from_this();
    std::deque<std::pair<int, const impl_t::query_t*>> signQueue;

    for (int i = 0; i < nsigs; ++i) {
        std::string domain;
        auto sigDomain = dkim_sig_getdomain(sigs[i]);
        while (sigDomain && *sigDomain != '\0') {
            domain.push_back(::tolower(*sigDomain++));
        }

        auto it = impl->signatures_.find(domain);
        if (it == impl->signatures_.end()) {
            dkim_sig_ignore(sigs[i]);
            continue;
        }
        const auto& sign = it->second;

        std::string selector;
        auto sigSelector = dkim_sig_getselector(sigs[i]);
        while (sigSelector && *sigSelector != '\0') {
            selector.push_back(::tolower(*sigSelector++));
        }
        if (selector != sign.selector) {
            dkim_sig_ignore(sigs[i]);
            continue;
        }

        if (!impl->primary_signature_domain_.empty() && domain == impl->primary_signature_domain_) {
            signQueue.emplace_front(i, &sign.dnsQuery);
        } else {
            signQueue.emplace_back(i, &sign.dnsQuery);
        }
    }

    if (signQueue.empty()) {
        return DKIM_CBSTAT_CONTINUE;
    }

    int successInd = -1;
    for (const auto& elem : signQueue) {
        dkim_set_user_context(dkim, const_cast<impl_t::query_t*>(elem.second));
        if (sig_process(dkim, sigs[elem.first])) {
            successInd = elem.first;
            break;
        }
    }

    if (successInd == -1) {
        successInd = signQueue.front().first;
    }

    for (const auto& elem : signQueue) {
        if (elem.first != successInd) {
            dkim_sig_ignore(sigs[elem.first]);
        }
    }

    dkim_set_user_context(dkim, nullptr);

    return DKIM_CBSTAT_CONTINUE;
}

} // namespace

const char* dkim_check::status(dkim_status s)
{
    switch (s)
    {
        case pass:
            return "pass";
        case fail:
            return "fail";
        case neutral:
            return "neutral";
        case none:
        default:
            return "none";
    }
}

void dkim_check::start(const input& p, const resolver_options& resolver_options, Handler handler, std::chrono::milliseconds timeout) {
    impl_ = boost::make_shared<impl>(ios, p, resolver_options);
    impl_->start(std::move(handler), timeout);
}

void dkim_check::stop() {
    if (impl_) {
        impl_->strand.post(std::bind(&impl::stop, impl_));
        impl_.reset();
    }
}

bool dkim_check::is_inprogress() const
{
    return impl_ && !impl_->done_;
}

void dkim_check::impl::stop() {
    resolver.cancel();
    timer.cancel();
    done_ = true;
}

void dkim_check::impl::complete(const Output& result) {
    stop();
    if (handler) {
        swap_and_call(handler, result);
    }
}

void dkim_check::impl::startTimer(std::chrono::milliseconds timeout) {
    timer.expires_from_now(timeout);
    timer.async_wait(strand.wrap(
        std::bind(&impl::handleTimer, shared_from_this(), _1))
    );
}

void dkim_check::impl::handleTimer(boost::system::error_code ec) {
    if (ec == boost::asio::error::operation_aborted || done_) {
        return;
    }
    if (p_.logger) {
        p_.logger("verify error: request timed out");
    }
    return complete(dkim_check::Output{neutral, "", "", std::list<std::string>{}});
}

void dkim_check::impl::cont() {
    try {
        verify();
    } catch (const std::exception& exp) {
        if (p_.logger) {
            p_.logger(std::string("verify error: ") + exp.what());
        }
        std::string identity;
        if (!primary_signature_domain_.empty()) {
            identity = "@" + primary_signature_domain_;
        }
        return complete(dkim_check::Output{neutral, std::move(identity), primary_signature_domain_, std::list<std::string>{}});
    }
}

void dkim_check::impl::verify() {
    if (done_) {
        throw std::runtime_error("request cancelled");
    }

    // Failed all DNS requests
    if (signatures_.empty()) {
        throw std::runtime_error("failed all DNS requests");
    }

    DKIM_STAT st;
    const uchar_t empty[] = {0};
    std::unique_ptr<DKIM, decltype(&dkim_free)> dkim(
        dkim_verify(lib1.instance(), empty, nullptr, &st),
        dkim_free);

    if (!dkim || st != DKIM_STAT_OK) {
        throw std::runtime_error(dkim_getresultstr(st));
    }

    addHeadersForCheck(dkim.get());

    void* ctx = this;
    dkim_set_user_context(dkim.get(), ctx);

    set_body(dkim.get(), p_.body.begin(), p_.body.end());
    st = dkim_eom(dkim.get(), nullptr);

    DKIM_SIGINFO* sig = dkim_getsignature(dkim.get());

    const char *dkim_error = dkim_geterror(dkim.get());
    if (dkim_error && p_.logger) {
        p_.logger("libopendkim error: " + std::string(dkim_error));
    }

    if (!sig) {
        throw std::runtime_error(dkim_getresultstr(st));
    }

    Output output;

    const char* sigInfoDomain = reinterpret_cast<const char *>(dkim_sig_getdomain(sig));
    if (sigInfoDomain) {
        output.domain = sigInfoDomain;
        output.domainList = {output.domain};
    }

    if (st != DKIM_STAT_OK && p_.logger) {
        p_.logger("verify error: " + std::string(dkim_getresultstr(st)));
        if (dkim_sig_getbh(sig) == DKIM_SIGBH_MISMATCH) {
            p_.logger("verify error: body hash mismatch");
        }
    }

    std::array<u_char, 256> identity;
    identity[0] = '\0';

    throw_if_error(dkim_sig_getidentity(dkim.get(), sig, identity.data(), identity.size()));

    output.identity.assign(identity.begin(), std::find(identity.begin(), identity.end(), '\0'));

    if (st == DKIM_STAT_OK) {
        output.status = pass;
    } else if (st == DKIM_STAT_BADSIG) {
        output.status = fail;
    } else {
        output.status = neutral;
    }

    return complete(output);
}

void dkim_check::impl::findSignatures() {
    auto headers = p_.header_storage->GetHeaders("dkim-signature");
    int minDepth = std::numeric_limits<int>::max();
    std::pair<std::string, signature> bestSignature;

    for (const auto& header : headers) {
        std::string domain;
        std::string selector;
        std::vector<std::string> headerList;

        NDkim::TSignatureParser parser(domain, selector, headerList);

        if (!parser.Parse(header.Value)) {
            continue;
        }

        int depth = find_least_common_domain_depth(domain, p_.from_domain);
        bool isBestSignature = depth > 0 && depth < minDepth;
        signature* sign;

        if (isBestSignature) {
            minDepth = depth;
            bestSignature.first = domain;
            sign = &bestSignature.second;
        } else if (signatures_.empty()) {
            sign = &signatures_[domain];
        } else {
            continue;
        }

        auto& dnsRequest = sign->dnsQuery.first;
        snprintf(dnsRequest.data(), dnsRequest.size(), "%s.%s.%s",
            selector.c_str(), DKIM_DNSKEYNAME, domain.c_str());

        sign->selector = std::move(selector);
        sign->header = {header.Name.begin(), header.Value.end()};
        sign->headerList = std::move(headerList);
        sign->orderNum = header.OrderNum;

        if (minDepth == 1 && !signatures_.empty()) {
            break;
        }
    }

    if (minDepth < std::numeric_limits<int>::max()) {
        primary_signature_domain_ = bestSignature.first;
        signatures_.insert(std::move(bestSignature));

    } else if (!signatures_.empty()) {
        primary_signature_domain_ = signatures_.begin()->first;
    }
}

void dkim_check::impl::addHeadersForCheck(DKIM* dkim) {
    std::unordered_set<std::string> needHeaders;
    std::vector<const signature*> signatures;

    for (const auto& elem : signatures_) {
        const auto& sign = elem.second;
        needHeaders.insert(sign.headerList.begin(), sign.headerList.end());
        signatures.push_back(&sign);
    }

    std::sort(signatures.begin(), signatures.end(), [](auto a, auto b) {
        return a->orderNum < b->orderNum;
    });

    signature fakeSignature;
    fakeSignature.orderNum = std::numeric_limits<unsigned int>::max();
    signatures.push_back(&fakeSignature);

    unsigned int limOrderNum = 0;
    for (auto sign : signatures) {
        for (const auto& headerName : needHeaders) {
            auto headers = p_.header_storage->GetHeaders(headerName);
            for (const auto& header : headers) {
                if (header.OrderNum < limOrderNum) {
                    continue;
                } else if (header.OrderNum >= sign->orderNum) {
                    break;
                }

                add_header(dkim, header.Name.begin(), header.Value.end());
            }
        }

        // Last (fake) signature
        if (sign->headerList.empty()) {
            break;
        }

        add_header(dkim, sign->header.begin(), sign->header.end());
        limOrderNum = sign->orderNum + 1;
    }

    throw_if_error(dkim_eoh(dkim));
}

void dkim_check::impl::start(dkim_check::Handler callback, std::chrono::milliseconds timeout)
{
    handler = std::move(callback);
    if (timeout != std::chrono::milliseconds::max()) {
        startTimer(timeout);
    }

    if (p_.logger) {
        auto count = p_.header_storage->Count("dkim-signature");
        p_.logger(std::to_string(count) + " signatures found");
    }

    findSignatures();

    if (signatures_.empty()) {
        if (p_.logger) {
            p_.logger("verify error: fail to find valid signature");
        }
        return complete(dkim_check::Output{neutral, "", "", std::list<std::string>{}});
    }

    auto it = signatures_.begin();
    resolver.async_resolve_txt(
        it->second.dnsQuery.first.data(),
        strand.wrap([it, impl = shared_from_this()](auto ec, auto resolveIter) {
            y_dkim_key_lookup_collect_helper(ec, resolveIter, impl, it);
        })
    );
}

namespace {

std::optional<DkimOptions::KeyEntry> get_domain_key(const DkimOptions::SignOptions::Keys& keys,
        const std::string& domain) {
    if (const auto element{keys.find(domain)}; element != keys.end()) {
        return element->second;
    }

    return {};
}

std::optional<DkimOptions::KeyEntry> find_subdomain_key(const DkimOptions::SignOptions::Keys& keys,
        const std::string& domain) {
    std::vector<boost::iterator_range<std::string::const_iterator>> labels;
    boost::split(labels, domain, boost::is_any_of("."));
    if (labels.size() < 2) {
        return {};
    }

    auto label{labels.cbegin()};
    const auto end{std::prev(std::prev(labels.cend()))};
    do {
        const auto key{get_domain_key(keys, {label->begin(), labels.back().end()})};
        if (key) {
            return key;
        }
    } while (label++ != end);

    return {};
}

bool parse_email(const std::string& email, std::string& local, std::string& domain) {
    auto at = email.find('@');
    if (at != std::string::npos) {
        local.assign(email.begin(), std::next(email.begin(), at));
        domain.assign(std::next(email.begin(), at + 1), email.end());
        std::transform(domain.begin(), domain.end(), domain.begin(), ::tolower);
        return true;
    }
    return false;
}

} // namespace {

void dkim_sign::start(const input& input, HttpClientPtr httpClientImpl, Handler handler) {
    pp = std::make_unique<struct input>(input);
    Context = boost::make_shared<TContext>(pp->session_id, pp->envelope_id, gconfig->clusterName, gconfig->hostName);
    callback = std::move(handler);
    httpClient = std::move(httpClientImpl);
    find_key([self = shared_from_this(), this]
        (DkimOptions::KeyEntry key, bool hasError, std::string description) {
            if (!hasError && !key.domain.empty()) {
                return try_sign(key);
            } else {
                return complete(description, {"", "", hasError});
            }
        });
}

void dkim_sign::find_key(KeyHandler handler) {
    std::string name, domain;
    if (!parse_email(pp->from, name, domain)) {
        return handler({}, true, "failed to parse email: " + pp->from);
    }

    auto key{get_domain_key(opt.keys, domain)};
    if (key) {
        pp->logger("dkim key got from file system for domain: " + key->domain);
        return handler(std::move(*key), false, {});
    }

    if (opt.use_fouras) {
        auto fourasCallback{[=, handler = std::move(handler)](
                auto ec, auto fourasKey) {
            if (!ec) {
                DkimOptions::KeyEntry key{
                    std::move(fourasKey.Selector),
                    std::move(fourasKey.Domain),
                    std::move(fourasKey.SecretKey)};
                pp->logger("dkim key got from Fouras for domain: " + key.domain);
                return handler(std::move(key), false, {});
            }

            auto subdomainKey{find_subdomain_key(opt.keys, domain)};
            if (subdomainKey) {
                pp->logger("failed to get dkim key from Fouras for domain: " + domain +
                        ", got from file system for subdomain: " + subdomainKey->domain);
                return handler(std::move(*subdomainKey), false, {});
            }

            std::string errorExtension{"; the specified domain not found in dkim key list: " + domain};
            if (ec == NFouras::EC_DOMAIN_NOT_FOUND || ec == NFouras::EC_DISABLED) {
                return handler({}, false, "key for " + domain + " not received from Fouras [" + ec.message() +
                        "]" + errorExtension);
            }

            return handler({}, true, "failed to get key for " + domain + " from Fouras" + errorExtension);
        }};

        auto client{std::make_shared<NFouras::TClient>(httpClient, domain, Context,
            ios.wrap(std::move(fourasCallback)))};
        yplatform::spawn(ios.get_executor(), std::move(client));
    } else {
        auto subdomainKey{find_subdomain_key(opt.keys, domain)};
        if (subdomainKey) {
            pp->logger("dkim key got from file system for subdomain: " + subdomainKey->domain);
            return handler(std::move(*subdomainKey), false, {});
        } else {
            return handler({}, false, "the specified domain not found in dkim key list: " + domain);
        }
    }
}

void dkim_sign::try_sign(const DkimOptions::KeyEntry& key) {
    std::string error;
    Output res;

    try {
        res = sign(key);
    } catch (const std::exception& exp) {
        error = exp.what();
        res.hasError = true;
    }

    complete(std::move(error), res);
}

dkim_sign::Output dkim_sign::sign(const DkimOptions::KeyEntry& key) {
    DKIM_STAT st;
    const uchar_t empty[] = {0};
    std::unique_ptr<DKIM, decltype(&dkim_free)> dkim(
        ::dkim_sign(
            lib2.instance(),
            empty,
            nullptr,
            const_cast<uchar_t*>(reinterpret_cast<const uchar_t*>(key.secretkey.c_str())), // broken OpenDKIM- API
            reinterpret_cast<const uchar_t*>(key.selector.c_str()),
            reinterpret_cast<const uchar_t*>(key.domain.c_str()),
            DKIM_CANON_RELAXED,
            DKIM_CANON_RELAXED,
            -1,
            -1,
            &st),
        dkim_free);

    if (!dkim || st != DKIM_STAT_OK) {
        throw std::runtime_error(dkim_getresultstr(st));
    }

    auto& headerNames = pp->header_storage->GetUniqueHeaderNames();
    for (const auto& headerName : headerNames) {
        if ((!opt.sign_hdrs.empty() && opt.sign_hdrs.count(headerName) == 0)
            || opt.skip_hdrs.count(headerName) > 0)
        {
            continue;
        }

        auto headers = pp->header_storage->GetHeaders(headerName);
        for (const auto& header : headers) {
            add_header(dkim.get(), header.Name.begin(), header.Value.end());
        }
    }

    throw_if_error(dkim_eoh(dkim.get()));

    set_body(dkim.get(), pp->body.begin(), pp->body.end());

    std::array<uchar_t, 1024> sig;

    st = dkim_eom(dkim.get(), nullptr);
    if (st == DKIM_STAT_OK) {
        st = dkim_getsighdr(dkim.get(), sig.data(), sig.size(), 0);
    }

    const char *dkim_error = dkim_geterror(dkim.get());
    if (dkim_error) {
        throw std::runtime_error(dkim_error);
    }
    throw_if_error(st);

    Output output;

    output.header = "DKIM-Signature: "
        + std::string(sig.begin(), std::find(sig.begin(), sig.end(), 0)) + "\r\n";
    output.identity = "@" + key.domain;

    return output;
}

void dkim_sign::complete(std::string error, const Output& output) {
    if (!error.empty()) {
        error = "dkim: " + error;
    }
    callback(error, output);
}

void find_domain(const std::string &header, std::string &result)
{
    uchar_t *user = NULL, *domain = NULL;
    std::vector<uchar_t> buffer(header.begin(), header.end());
    buffer.push_back('\0');
    result.clear();
    if (dkim_mail_parse(&buffer[0], &user, &domain) == 0 && domain != NULL)
        std::copy(domain, std::find(domain, &buffer.back(), '\0'),
                std::back_inserter(result));
}

int find_least_common_domain_depth(const std::string& signatureDomain, const std::string& fromDomain) {
    if (fromDomain.size() < signatureDomain.size()) {
        return 0;
    }

    std::vector<std::string> signatureParts, fromParts;

    boost::split(signatureParts, signatureDomain, boost::is_any_of("."));
    boost::split(fromParts, fromDomain, boost::is_any_of("."));

    if (fromParts.size() < signatureParts.size()) {
        return 0;
    }

    for (auto sit = signatureParts.crbegin(), fit = fromParts.crbegin(); sit != signatureParts.crend(); ++sit, ++fit) {
        if (*sit != *fit) {
            return 0;
        }
    }

    return fromParts.size() - signatureParts.size() + 1;
}

}   // namespace NNwSmtp
