#pragma once

#include "dkim_options.h"

#include <mail/nwsmtp/src/context.h>
#include <mail/nwsmtp/src/header_storage.h>
#include <mail/nwsmtp/src/types.h>

#include <nwsmtp/resolver.h>

#include <ymod_httpclient/call.h>

#include <boost/function.hpp>
#include <boost/asio.hpp>

#include <chrono>
#include <list>
#include <memory>

namespace NNwSmtp {

typedef boost::function<void(const std::string&)> dkim_logger_t;

class dkim_check
{
  public:
    struct impl;

    using THeaderStorage = NNwSmtp::THeaderStorageImpl;
    using TBufferConstIteratorRange = NNwSmtp::TBufferRange;

    enum dkim_status
    {
        pass,
        neutral,
        fail,
        none
    };

    struct input
    {
        input(
            const std::string& fromDomain,
            std::shared_ptr<THeaderStorage> headerStorage,
            TBufferConstIteratorRange messageBody,
            const dkim_logger_t& logger
        )
            : from_domain(fromDomain)
            , header_storage(std::move(headerStorage))
            , body(messageBody)
            , logger(logger)
        {}

        std::string from_domain;
        std::shared_ptr<THeaderStorage> header_storage;
        TBufferConstIteratorRange body;
        dkim_logger_t logger;
    };

    struct Output {
        dkim_status status;
        std::string identity;
        std::string domain;
        std::list<std::string> domainList;
    };

    using Handler = std::function<void (const Output&)>;

    dkim_check(boost::asio::io_service& ios)
        : ios(ios)
    {}

    void start(
        const input& p,
        const resolver_options& resolver_options,
        Handler handler,
        std::chrono::milliseconds timeout = std::chrono::milliseconds::max()
    );
    void stop();

    bool is_inprogress() const;

    static const char* status(dkim_status s);

  private:
    boost::asio::io_service& ios;
    boost::shared_ptr<impl> impl_;
};

class dkim_sign : public std::enable_shared_from_this<dkim_sign> {
public:
    using THeaderStorage = NNwSmtp::THeaderStorageImpl;
    using TBufferConstIteratorRange = NNwSmtp::TBufferRange;

    struct input
    {
        input(
            const std::string& from,
            std::shared_ptr<THeaderStorageImpl> headerStorage,
            TBufferRange messageBody,
            const std::string& sessionId,
            const std::string& envelopeId,
            dkim_logger_t logger
        )
            : from(from)
            , header_storage(std::move(headerStorage))
            , body(messageBody)
            , session_id(sessionId)
            , envelope_id(envelopeId)
            , logger(std::move(logger))
        {}

        std::string from;
        std::shared_ptr<THeaderStorageImpl> header_storage;
        TBufferRange body;
        std::string session_id;
        std::string envelope_id;
        dkim_logger_t logger;
    };

    struct Output {
        std::string header;
        std::string identity;
        bool hasError = false;
    };

    using Handler = std::function<void (const std::string& error, const Output&)>;
    using HttpClientPtr = std::shared_ptr<ymod_httpclient::cluster_call>;

    dkim_sign(boost::asio::io_context& ios, const DkimOptions::SignOptions& opt)
        : ios(ios)
        , opt(opt)
    {}

    void start(const input& p, HttpClientPtr httpClient, Handler handler);

private:
    using KeyHandler = std::function<void(DkimOptions::KeyEntry key, bool hasError, std::string description)>;

    void find_key(KeyHandler handler);

    void try_sign(const DkimOptions::KeyEntry& key);
    Output sign(const DkimOptions::KeyEntry& key);

    void complete(std::string error, const Output& output);

private:
    boost::asio::io_context& ios;
    const DkimOptions::SignOptions& opt;
    std::unique_ptr<input> pp;
    TContextPtr Context;
    HttpClientPtr httpClient;
    Handler callback;
};

void find_domain(const std::string& header, std::string &result);

int find_least_common_domain_depth(const std::string& signatureDomain, const std::string& fromDomain);

void init_dkim();

}   // namespace NNwSmtp
