#include "dkim_module.h"

#include <ymod_httpclient/cluster_client.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>

namespace NNwSmtp {

namespace detail {

template <typename Ctx, typename Impl>
struct HandlerWrapper {
    HandlerWrapper(Ctx ctx, Impl&& impl)
        : ctx(ctx)
        , impl(std::forward<Impl>(impl))
    {}

    template <typename... Args>
    void operator()(Args&& ...args) {
        impl(std::forward<Args>(args)...);
    }

    template <typename... Args>
    void operator()(Args&& ...args) const {
        impl(std::forward<Args>(args)...);
    }

    Ctx ctx;
    typename std::decay<Impl>::type impl;
};

template <typename Ctx, typename Impl>
inline HandlerWrapper<Ctx, Impl> wrap_handler(Ctx ctx, Impl&& impl) {
    return HandlerWrapper<Ctx, Impl>(ctx, std::forward<Impl>(impl));
}

}   // namespace detail

Dkim::Dkim(yplatform::reactor& reactor)
    : reactor(reactor)
{}

void Dkim::init(const yplatform::ptree& pt) {
    options = std::make_shared<DkimOptions>(pt);
    if (options->signOpts.use_fouras) {
        auto fourasReactor = yplatform::find_reactor(pt.get<std::string>("fouras.reactor"));
        httpClient =
            std::make_shared<ymod_httpclient::cluster_client>(*fourasReactor, pt.get_child("fouras.http_client"));
    }
    init_dkim();
}

void Dkim::sign(const dkim_sign::input& data, dkim_sign::Handler handler) {
    auto ios = reactor.io();
    auto sign = std::make_shared<dkim_sign>(*ios, options->signOpts);
    ios->post(std::bind(
        &dkim_sign::start,
        sign,
        data,
        httpClient,
        detail::wrap_handler(shared_from_this(), std::move(handler)))
    );
}

std::shared_ptr<dkim_check> Dkim::check(const dkim_check::input& data, const resolver_options& resolver_options, dkim_check::Handler handler) {
    auto ios = reactor.io();
    auto check = std::make_shared<dkim_check>(*ios);
    ios->post(std::bind(
        &dkim_check::start,
        check,
        data,
        resolver_options,
        detail::wrap_handler(shared_from_this(), std::move(handler)),
        std::chrono::seconds(options->timeout))
    );
    return check;
}

auto FindDkimService() {
    return yplatform::find<NNwSmtp::Dkim, std::shared_ptr>("dkim");
}

DkimOptions::dkim_mode GetDkimMode() {
    return FindDkimService()->getOptions().mode;
}

bool AllowSkipDkimSign() {
    return FindDkimService()->getOptions().signOpts.allow_skip_sign;
}

void AsyncDkimCheck(
    const dkim_check::input& data,
    const resolver_options& resolverOpts,
    dkim_check::Handler handler)
{
    FindDkimService()->check(data, resolverOpts, handler);
}

void AsyncDkimSign(const dkim_sign::input& input, dkim_sign::Handler handler) {
    FindDkimService()->sign(input, handler);
}

}   // namespace NNwSmtp


DEFINE_SERVICE_OBJECT(NNwSmtp::Dkim)
