#include "dkim_options.h"
#include "../file_converter.h"

#define _Bool bool
#include <contrib/libs/libopendkim/libopendkim/dkim.h>
#undef _Bool

#include <boost/property_tree/json_parser.hpp>
#include <boost/algorithm/string/case_conv.hpp>
#include <boost/range/iterator_range.hpp>
#include <boost/range/algorithm/equal_range.hpp>
#include <boost/range/as_literal.hpp>
#include <boost/filesystem.hpp>
#include <boost/spirit/include/qi.hpp>

#include <algorithm>
#include <iterator>
#include <fstream>
#include <vector>
#include <string>
#include <chrono>
#include <thread>
#include <map>

using namespace NNwSmtp;

using KeyEntry = DkimOptions::KeyEntry;
using Keys = DkimOptions::SignOptions::Keys;

typedef DkimOptions::dkim_mode dkim_mode_t;

namespace {

template <class Iterator>
dkim_mode_t parse_mode(Iterator beg, Iterator end)
{
    boost::iterator_range<Iterator> r(beg, end);
    if (boost::equal(r, boost::as_literal("none")))
        return DkimOptions::none;
    if (boost::equal(r, boost::as_literal("sign")))
        return DkimOptions::sign;
    if (boost::equal(r, boost::as_literal("signverify")))
        return DkimOptions::signverify;
    if (boost::equal(r, boost::as_literal("verify")))
        return DkimOptions::verify;
    else
    {
        std::ostringstream os;
        os << "invalid dkim mode: " << r;
        throw std::invalid_argument(os.str());
    }
}

class line
{
    std::string s_;
public:
    friend std::istream & operator>>(std::istream & is, line& l)
    {
        std::ws(is);
        return std::getline(is, l.s_);
    }
    operator std::string() const { return s_; }
};

template <class Iterator>
Keys parse_keys(Iterator beg, Iterator end) {
    Keys res;

    using boost::spirit::lit;
    using boost::spirit::lexeme;
    using boost::spirit::raw;
    using boost::spirit::omit;
    using boost::spirit::qi::char_;
    using boost::spirit::qi::space;

    while (beg != end)
    {
        std::string l = *beg++;
        if (l.empty())
            break;

        typedef std::string::const_iterator iterator_t;
        boost::iterator_range<iterator_t> domain;
        boost::iterator_range<iterator_t> selector;
        boost::iterator_range<iterator_t> path;

        iterator_t lbeg = l.begin();
        iterator_t lend = l.end();

        if (!boost::spirit::qi::parse(
                    lbeg,
                    lend,
                    omit[*space >> +(~char_(' ')) >> +char_(' ')]
                    >> raw[lexeme[+(~char_(':'))]] >> ':'
                    >> raw[lexeme[+(~char_(':'))]] >> ':'
                    >> raw[lexeme[+char_]],
                    domain,
                    selector,
                    path))
        {
            std::ostringstream os;
            os << "failed to parse dkim key entry: " << l;
            throw std::invalid_argument(os.str());
        }

        KeyEntry key;
        key.domain.assign(domain.begin(), domain.end());
        key.selector.assign(selector.begin(), selector.end());

        std::string secretkey_filename{path.begin(), path.end()};

        std::ifstream file(secretkey_filename.c_str());
        if (!file)
        {
            std::ostringstream os;
            os << "failed to open dkim secretkey: " << secretkey_filename;
            throw std::invalid_argument(os.str());
        }

        file.unsetf(std::ios_base::skipws);
        std::copy(std::istream_iterator<char>(file),
                std::istream_iterator<char>(),
                std::back_inserter(key.secretkey));

        res.emplace(std::string(domain.begin(), domain.end()), std::move(key));
    }
    return res;
}

struct ParseThreadData {
    std::vector<line> linesToParse;
    Keys keys;
    std::exception_ptr error;
};

template <class Iterator>
void parse_keys(Iterator beg, Iterator end, Keys& dst, unsigned int threadCount) {
    std::map<unsigned int, ParseThreadData> threadData;
    unsigned int linesCounter = 0;
    // read map : domain -> path/to/key
    while (beg != end) {
        threadData[linesCounter++ % threadCount].linesToParse.emplace_back(*beg++);
    }
    // multithreaded parsing
    std::vector<std::thread> threads;
    for (auto& data : threadData) {
        threads.emplace_back([&data]() {
            auto& result = data.second;
            try {
                result.keys = parse_keys(result.linesToParse.begin(), result.linesToParse.end());
            } catch (...) {
                result.error = std::current_exception();
            }
        });
    }
    for (auto& t : threads) {
        t.join();
    }
    // merge results
    for (const auto& result : threadData) {
        if (result.second.error) {
            std::rethrow_exception(result.second.error);
        }
        dst.insert(result.second.keys.begin(), result.second.keys.end());
    }
}

void ParsePrimaryKeys(const std::string& filename, Keys& dst) {
    boost::property_tree::ptree tree;
    boost::property_tree::read_json(filename, tree);

    for (auto& [name, pt] : tree) {
        std::string domain;
        KeyEntry key;

        domain = pt.get<std::string>("domain");
        key.domain = domain;
        key.selector = pt.get<std::string>("selector");
        key.secretkey = pt.get<std::string>("key");

        dst.emplace(std::move(domain), std::move(key));
    }
}

} // namespace {

namespace NNwSmtp {

struct SignOptionsParser {
    DkimOptions::SignOptions& options;
    unsigned int threadCount;

    explicit SignOptionsParser(DkimOptions::SignOptions& opts, unsigned int threadCount = 16)
        : options(opts)
        , threadCount(threadCount)
    {}

    void parseFromPtree(const ptree& pt, dkim_mode_t mode, bool forceLoad = false);
};


DkimOptions::DkimOptions(const ptree& pt) { read(pt); }

void DkimOptions::read(const ptree& pt) {
    auto modeStr = pt.get<std::string>("mode");
    mode = parse_mode(modeStr.begin(), modeStr.end());

    threadsToParseKeys = pt.get<unsigned int>("threads_to_parse_keys", 16);
    bool forceLoad = pt.get<bool>("force_load", false);
    SignOptionsParser parser(signOpts, threadsToParseKeys);
    parser.parseFromPtree(pt, mode, forceLoad);

    timeout = pt.get<unsigned int>("timeout", 1);
}

void SignOptionsParser::parseFromPtree(const ptree& pt, dkim_mode_t mode, bool forceLoad) {
    if ((mode == DkimOptions::none || mode == DkimOptions::verify) && !forceLoad) {
        return;
    }

    options.allow_skip_sign = pt.get<bool>("allow_skip_sign_if_error", options.allow_skip_sign);
    options.use_fouras = pt.get("fouras.use", options.use_fouras);

    const auto keysPt = pt.get_child_optional("keys");
    if (keysPt) {
        // Parse keys
        if (keysPt->count("default")) {
            auto keysFilename = keysPt->get<std::string>("default");
            std::ifstream file(keysFilename, std::ios_base::in);
            if (!file) {
                throw std::runtime_error(std::string("failed to open keys file:") + keysFilename);
            }
            std::istream_iterator<line> beg(file);
            std::istream_iterator<line> end;
            options.keys.clear();
            parse_keys(beg, end, options.keys, threadCount);
        }
        // Parse yandex keys
        if (keysPt->count("yandex")) {
            auto yaKeysFile = keysPt->get<std::string>("yandex");
            std::ifstream file(yaKeysFile, std::ios_base::in);
            if (!file) {
                throw std::runtime_error(std::string("failed to open yandex keys file:") + yaKeysFile);
            }
            std::istream_iterator<line> beg(file);
            std::istream_iterator<line> end;
            parse_keys(beg, end, options.keys, threadCount);
        }
        // Parse keys in json format
        if (keysPt->count("primary")) {
            auto primaryKeysFile = keysPt->get<std::string>("primary");
            ParsePrimaryKeys(primaryKeysFile, options.keys);
        }
    }

    auto headersSigned = GeDkimDefaultSignedHeaders();
    for (auto& header : headersSigned) {
            options.sign_hdrs.insert(std::move(header));
    }

    auto range = pt.equal_range("sign_headers_list");
    for (auto header = range.first; header != range.second; ++header) {
        options.sign_hdrs.insert(boost::to_lower_copy(header->second.data()));
    }

    auto headersNotSigned = GeDkimDefaultNotSignedHeaders();
    for (auto& header : headersNotSigned) {
        options.skip_hdrs.insert(std::move(header));
    }
}

std::vector<std::string> GeDkimDefaultSignedHeaders() {
    std::vector<std::string> headers;
    for (const u_char** h = dkim_should_signhdrs; *h != nullptr; ++h) {
        std::string hdr(reinterpret_cast<const char*>(*h));
        boost::to_lower(hdr);
        headers.push_back(std::move(hdr));
    }
    return headers;
}

std::vector<std::string> GeDkimDefaultNotSignedHeaders() {
    std::vector<std::string> headers;
    for (const u_char** h = dkim_should_not_signhdrs; *h != nullptr; ++h) {
        std::string hdr(reinterpret_cast<const char*>(*h));
        boost::to_lower(hdr);
        headers.push_back(std::move(hdr));
    }
    return headers;
}

} // namespace NNwSmtp
