#include "signature_parser.h"

#include <cctype>

namespace NNwSmtp {
namespace NDkim {

TSignatureParser::TSignatureParser(std::string& domain, std::string& selector, TFields& fields)
    : Domain(domain)
    , Selector(selector)
    , Fields(fields)
{}

bool TSignatureParser::ParseSymbol(char c) {
    // Read tag name
    if (State == EState::WaitTagName) {
        // https://tools.ietf.org/html/rfc6376#section-3.2
        if (std::isalpha(c) || (!CurrentTag.empty() && (std::isdigit(c) || c == '_'))) {
            CurrentTag.push_back(std::tolower(c));
        } else if (c == '=') {
            State = EState::WaitTagsDelimiter;
        } else if (std::isspace(c)) {
            if (!CurrentTag.empty()) {
                State = EState::WaitTagValueDelimiter;
            }
        } else {
            return false;
        }

        if (State != EState::WaitTagName) {
            if (CurrentTag.empty()) {
                return false;
            }

            auto count = RequiredTags.erase(CurrentTag);

            // Domain tag found
            if (CurrentTag == "d" && count == 1) {
                Parser = std::make_unique<TDomainParser>(Domain);

            // Selector found
            } else if (CurrentTag == "s" && count == 1) {
                Parser = std::make_unique<TDomainParser>(Selector);

            // Header list found
            } else if (CurrentTag == "h" && count == 1) {
                Parser = std::make_unique<TFieldsParser>(Fields);
            }

            CurrentTag.clear();
        }

    // Read delimiter '='
    } else if (State == EState::WaitTagValueDelimiter) {
        if (c == '=') {
            State = EState::WaitTagsDelimiter;
        } else if (!std::isspace(c)) {
            return false;
        }

    // Read tag value
    } else if (State == EState::WaitTagsDelimiter) {
        // Found end of tag value
        if (c == ';') {
            State = EState::WaitTagName;

            if (Parser && Parser->Finalize() != ITagParser::EState::Ok) {
                return false;
            }

            Parser.reset();
        } else if (Parser && Parser->Parse(c) == ITagParser::EState::Error) {
            return false;
        }
    }

    return true;
}

void TSignatureParser::Reset() {
    State = EState::WaitTagName;
    Parser.reset();

    CurrentTag.clear();

    // https://tools.ietf.org/html/rfc6376#section-3.5
    RequiredTags = {"a", "b", "bh", "d", "h", "s", "v"};
}

TDomainParser::TDomainParser(std::string& domain)
    : State(EState::Init)
    , Domain(domain)
    , PrevIsSpace(false)
{
    Domain.clear();
}

TDomainParser::EState TDomainParser::Parse(char c) {
    if (State == EState::Error) {
        return State;
    }

    if (std::isspace(c)) {
        if (State == EState::Ok) {
            PrevIsSpace = true;
        }

        return State;
    }

    if (PrevIsSpace) {
        State = EState::Error;
        return State;
    }

    State = EState::Ok;
    Domain.push_back(std::tolower(c));

    return State;
}

TDomainParser::EState TDomainParser::Finalize() {
    if (State == EState::Init) {
        State = EState::Error;
    }

    return State;
}

TFieldsParser::TFieldsParser(TFields& fields)
    : State(EState::Init)
    , Fields(fields)
    , PrevSpaceCount(0)
{
    Fields.clear();
    Fields.resize(1);
}

TFieldsParser::EState TFieldsParser::Parse(char c) {
    if (State == EState::Error) {
        return State;
    }

    if (std::isspace(c)) {
        if (State == EState::Ok) {
            ++PrevSpaceCount;
        }

        return State;
    }

    auto& field = Fields.back();

    if (c == ':') {
        if (field.empty()) {
            State = EState::Error;
        } else {
            Fields.resize(Fields.size() + 1);
        }

        return State;
    }

    if (PrevSpaceCount > 0 && !field.empty()) {
        field.append(PrevSpaceCount, ' ');
    }

    State = EState::Ok;
    PrevSpaceCount = 0;
    field.push_back(std::tolower(c));

    return State;
}

TFieldsParser::EState TFieldsParser::Finalize() {
    if (State == EState::Init || (State == EState::Ok && Fields.back().empty())) {
        State = EState::Error;
    }

    return State;
}

}   // namespace NDkim
}   // namespace NNwSmtp
