#pragma once

#include <memory>
#include <string>
#include <unordered_set>
#include <vector>

namespace NNwSmtp {
namespace NDkim {

class ITagParser {
public:
    enum class EState {
        Init,
        Ok,
        Error
    };

    ITagParser() = default;
    virtual ~ITagParser() = default;

    virtual EState Parse(char c) = 0;
    virtual EState Finalize() = 0;
};

class TSignatureParser {
public:
    enum class EState {
        WaitTagName,
        WaitTagValueDelimiter,
        WaitTagsDelimiter
    };

    using TFields = std::vector<std::string>;

   /**
     * @param domain The string that will contain the value of the tag d in lowercase.
     * @param selector The string that will contain the value of the tag s in lowercase.
     * @param fields The vector that will contain of signed header fields from the tag h in lowercase.
     *
     * It's guaranteed only that the values won't be empty and domain with selector won't contain whitespace characters.
    */
    TSignatureParser(std::string& domain, std::string& selector, TFields& fields);

    template <typename TBuffer>
    bool Parse(const TBuffer& headerValue) {
        Reset();

        for (const auto c : headerValue) {
            if (!ParseSymbol(c)) {
                return false;
            }
        }

        if (Parser && Parser->Finalize() != ITagParser::EState::Ok) {
            return false;
        }

        return RequiredTags.empty();
    }

private:
    bool ParseSymbol(char c);
    void Reset();

private:
    std::string& Domain;
    std::string& Selector;
    TFields& Fields;

    EState State;
    std::unique_ptr<ITagParser> Parser;

    std::string CurrentTag;
    std::unordered_set<std::string> RequiredTags;
};

class TDomainParser : public ITagParser {
public:
    explicit TDomainParser(std::string& domain);
    EState Parse(char c) final;
    EState Finalize() final;
private:
    EState State;
    std::string& Domain;
    bool PrevIsSpace;
};

class TFieldsParser : public ITagParser {
public:
    using TFields = std::vector<std::string>;

    explicit TFieldsParser(TFields& fields);
    EState Parse(char c) final;
    EState Finalize() final;
private:
    EState State;
    TFields& Fields;
    unsigned int PrevSpaceCount;
};

}   // namespace NDkim
}   // namespace NNwSmtp
