#include "parser.h"
#include "exceptions.h"
#include "util.h"

#include <libopendmarc/dmarc.h>

namespace NNwSmtp::NOpenDmarc {

TDmarcParser::TDmarcParser(const DMARC_POLICY_T* pctx)
    : Pctx(pctx) {}

TDmarcParser TDmarcParser::Parse(
    DMARC_POLICY_T* pctx,
    const std::string& fromDomain,
    const std::string& record)
{
    auto status = opendmarc_policy_parse_dmarc(
        pctx,
        ToUchar(fromDomain),
        ToUchar(record));

    CheckStatus(status);
    return TDmarcParser(pctx);
}

TRecordPtr TDmarcParser::MakeRecord() const {
    auto record = std::make_unique<TRecord>();
    record->DomainPolicy = GetDomainPolicy();
    record->SubDomainPolicy = GetSubDomainPolicy();
    record->Adkim = GetAdkim();
    record->Aspf = GetAspf();
    record->Pct = GetPct();
    record->FailureOpts = GetFailureOptions();
    record->Rua = GetRua();
    return record;
}

EDomainPolicy TDmarcParser::GetDomainPolicy() const {
    int p;
    CheckStatus(opendmarc_policy_fetch_p(Pctx, &p));
    return ToDomainPolicy(p);
}

EDomainPolicy TDmarcParser::GetSubDomainPolicy() const {
    int sp;
    CheckStatus(opendmarc_policy_fetch_sp(Pctx, &sp));
    return ToDomainPolicy(sp);
}

EAlignMode TDmarcParser::GetAdkim() const {
    int adkim;
    CheckStatus(opendmarc_policy_fetch_adkim(Pctx, &adkim));
    return ToAlignMode(adkim);
};

EAlignMode TDmarcParser::GetAspf() const {
    int aspf;
    CheckStatus(opendmarc_policy_fetch_aspf(Pctx, &aspf));
    return ToAlignMode(aspf);
};

uint32_t TDmarcParser::GetPct() const {
    int pct;
    CheckStatus(opendmarc_policy_fetch_pct(Pctx, &pct));
    if (pct < 0 || pct > 100) {
        throw TDmarcException("Percentage is out of range");
    }
    return static_cast<uint32_t>(pct);
}

EFailureOptions TDmarcParser::GetFailureOptions() const {
    int fo;
    CheckStatus(opendmarc_policy_fetch_fo(Pctx, &fo));
    return ToFailureOptions(fo);
}

std::vector<std::string> TDmarcParser::GetRua() const {
    std::vector<std::string> rua;
    const u_char* const* rawRua;
    if ((rawRua = opendmarc_policy_fetch_rua_no_network(Pctx))) {
        for (size_t idx = 0; rawRua[idx] != nullptr; ++idx) {
            // We think it's ok
            rua.emplace_back(reinterpret_cast<const char*>(rawRua[idx]));
        }
    }
    return rua;
}

}  // namespace NNwSmtp::NOpenDmarc
