#include "policy_manager.h"
#include "util.h"
#include "exceptions.h"

namespace NNwSmtp::NOpenDmarc {

TPolicyManager::TPolicyManager(
    DMARC_POLICY_T* pctx,
    const std::string& fromDomain,
    const std::string& txtRecord,
    const std::string& organizationalDomain
)
    : Pctx(pctx)
{
    auto organizationalDomainPtr = ToUchar(organizationalDomain);
    if (organizationalDomain.empty()) {
        organizationalDomainPtr = nullptr;
    }

    auto status = opendmarc_policy_store_dmarc(
        Pctx,
        ToUchar(txtRecord),
        ToUchar(fromDomain),
        organizationalDomainPtr);

    CheckStatus(status);
}

// Optional
void TPolicyManager::SetSpf(const std::string& fromDomain, EOutcome outcome, ESpfOrigin origin) {
    auto status = opendmarc_policy_store_spf(
        Pctx,
        ToUchar(fromDomain),
        ToRaw(outcome),
        ToRaw(origin),
        nullptr);
    CheckStatus(status);
}

void TPolicyManager::SetDkim(const std::string& dEqualDomain, EOutcome outcome) {
    // one or many for every dkim outcome.
    // domain from d= record of the dkim signature
    auto status = opendmarc_policy_store_dkim(
        Pctx,
        ToUchar(dEqualDomain),
        ToRaw(outcome),
        nullptr);
    CheckStatus(status);
}

EDmarcPolicy TPolicyManager::AdvisePolicy() {
    auto status = opendmarc_get_policy_to_enforce(Pctx);
    if ((status == DMARC_PARSE_ERROR_NULL_CTX) ||
        (status == DMARC_FROM_DOMAIN_ABSENT))
    {
        throw TDmarcParseError(status);
    }
    return ToDmarcPolicy(status);
}

TAlignments TPolicyManager::GetAlignments() const {
    int adkim;
    int aspf;
    CheckStatus(opendmarc_policy_fetch_alignment(Pctx, &adkim, &aspf));
    return {ToAlignment(adkim), ToAlignment(aspf)};
}

}  // namespace NNwSmtp::NOpenDmarc
