#pragma once

#include <util/generic/is_in.h>

#include <mutex>
#include <shared_mutex>
#include <set>

namespace NNwSmtp {

class TDomainsSet {
public:
    using TContainer = std::set<std::string>;

    template <typename TIterator>
    void Load(TIterator first, TIterator last) {
        for (; first != last; ++first) {
            LoadTo(first->second.data(), Domains);
        }
    }

    bool Contains(const std::string& domain) const {
        return IsIn(Domains, domain);
    }

    bool Empty() const {
        return Domains.empty();
    }

private:
    static void LoadTo(const std::string& src, TContainer& dst);

private:
    TContainer Domains;
};

class TThreadSafeDomainsSet {
public:
    using TReadLock = std::shared_lock<std::shared_timed_mutex>;
    using TWriteLock = std::unique_lock<std::shared_timed_mutex>;

    template <typename TIterator>
    void Load(TIterator first, TIterator last) {
        TDomainsSet newDomains;
        newDomains.Load(first, last);
        TWriteLock lock(Mutex);
        std::swap(newDomains, Domains);
    }

    bool Contains(const std::string& domain) const {
        TReadLock lock(Mutex);
        return Domains.Contains(domain);
    }

    bool Empty() const {
        TReadLock lock(Mutex);
        return Domains.Empty();
    }

private:
    TDomainsSet Domains;
    mutable std::shared_timed_mutex Mutex;
};

}   // namespace NNwSmtp
