#pragma once

#include <mail/nwsmtp/src/delivery/async/error_code.h>
#include <mail/nwsmtp/src/mailfrom/error_code.h>
#include <mail/nwsmtp/src/rcpt_to/error_code.h>
#include <mail/nwsmtp/src/smtp_client/error_code.h>

#include <boost/system/error_code.hpp>

#include <iostream>

namespace NNwSmtp::NSmtp {

enum class EError {
    Ok,
    TempFail,
    Discarded,
    Rejected
};

} // namespace NNwSmtp::NSmtp

namespace boost::system {

template <>
struct is_error_code_enum<NNwSmtp::NSmtp::EError> : std::true_type {};

} // namespace boost::system

namespace NNwSmtp::NSmtp {

class TErrorCategory final : public boost::system::error_category {

public:
    const char* name() const noexcept override {
        return "nwsmtp_smtp";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::Ok:
                return "Ok";
            case EError::TempFail:
                return "Delivery temporarily failed";
            case EError::Discarded:
                return "Delivery discarded";
            case EError::Rejected:
                return "Delivery rejected";
        }
        return "Unknown error code: " + std::to_string(value);
    }

    bool equivalent(const boost::system::error_code& code, int condition) const noexcept override {
        using namespace NNwSmtp;
        switch (EError(condition)) {
            case EError::Discarded: {
                return code == NRcptTo::EError::Discard
                    || code == NAsyncDlv::EError::MaliciousDiscarded
                    || code == NAsyncDlv::EError::AvirDiscarded
                    || code == NAsyncDlv::EError::SpamDiscarded;
            }
            case EError::Rejected: {
                return code == NAsyncDlv::EError::TooManyHeadersReject
                    || code == NAsyncDlv::EError::DeliveryRejected
                    || code == NAsyncDlv::EError::SenderIdRejected
                    || code == NAsyncDlv::EError::MaliciousRejected
                    || code == NAsyncDlv::EError::RpopAuthRejected
                    || code == NAsyncDlv::EError::AvirRejected
                    || code == SmtpClient::EError::Rejected
                    || code == NAsyncDlv::EError::SpamRejected
                    || code == NMailFromCommand::EError::MailAddrIsTooLong
                    || code == NMailFromCommand::EError::MailAddrIsInvalid
                    || code == NMailFromCommand::EError::MessageSizeIsInvalid
                    || code == NMailFromCommand::EError::MessageSizeExceedsLimit
                    || code == NMailFromCommand::EError::NotAuthorized
                    || code == NMailFromCommand::EError::AuthReject
                    || code == NMailFromCommand::EError::RejectAddrNotOwnedByAuthUser
                    || code == NMailFromCommand::EError::RejectAddrUserNotFound
                    || code == NMailFromCommand::EError::RejectRateSrv
                    || code == NRcptTo::EError::Reject
                    || code == NRcptTo::EError::RejectByRateSrv
                    || code == NRcptTo::EError::MaillistIsReadonly
                    || code == NRcptTo::EError::TooManyRecipients
                    || code == NAsyncDlv::EError::RfcFailRejected
                    || code == NAsyncDlv::EError::UrlRblRejected
                    || code == NAsyncDlv::EError::BadKarmaRejected
                    || code == NAsyncDlv::EError::MailLimitsRejected
                    || code == NAsyncDlv::EError::PddAdminKarmaRejected
                    || code == NAsyncDlv::EError::BouncesRejected
                    || code == NAsyncDlv::EError::SpamComplRejected
                    || code == NAsyncDlv::EError::RelayNotSet
                    || code == NAsyncDlv::EError::CycleDetected;
            }
            case EError::TempFail: {
                return code == NAsyncDlv::EError::DkimSignTempFail
                    || code == NAsyncDlv::EError::DeliveryTempFail
                    || code == NAsyncDlv::EError::RouteTempFail
                    || code == NAsyncDlv::EError::RpopAuthTempFail
                    || code == SmtpClient::EError::TempFail
                    || code == NAsyncDlv::EError::SoTempFail
                    || code == NMailFromCommand::EError::TempFail
                    || code == NRcptTo::EError::TempFail;
            }
            default:
                return boost::system::error_category::equivalent(code, condition);
        }
    }

};

inline const TErrorCategory& getErrorCategory() {
    static TErrorCategory errorCategory;
    return errorCategory;
}

inline auto make_error_code(EError ec) {
    return boost::system::error_code(static_cast<int>(ec), getErrorCategory());
}

inline auto make_error_condition(EError ec) {
    return boost::system::error_condition(static_cast<int>(ec), getErrorCategory());
}

} // namespace NNwSmtp::NSmtp
