#pragma once

#include <boost/algorithm/string.hpp>

#include <fstream>
#include <iterator>
#include <sstream>

namespace NNwSmtp {

namespace NConverters {

struct TEquivalent {
    template <typename T>
    T operator() (const T& value) const {
        return value;
    }
};

struct TToLower {
    template <typename T>
    T operator() (const T& value) const {
        return boost::to_lower_copy(value);
    }
};

}   // namespace NConverters

class TFileConverter {
public:
    template <typename TVector, typename TTrans = NConverters::TEquivalent>
    static void ToVector(const std::string& filename, TVector& vec, TTrans trans = TTrans()) {
        Translate(filename, std::back_inserter(vec), trans);
    }

    template <typename TSet, typename TTrans = NConverters::TEquivalent>
    static void ToSet(const std::string& filename, TSet& set, TTrans trans = TTrans()) {
        Translate(filename, std::inserter(set, set.begin()), trans);
    }

    template <typename TMap,
        typename TKeyTrans = NConverters::TEquivalent,
        typename TValueTrans = NConverters::TEquivalent>
    static void ToMap(
        const std::string& filename,
        TMap& map,
        TKeyTrans kTrans = TKeyTrans(),
        TValueTrans vTrans = TValueTrans())
    {
        auto transformer = [kTrans, vTrans] (const std::string& line) {
            std::stringstream ss(line);
            std::string keyStr, valueStr;
            ss >> keyStr >> valueStr;
            return std::make_pair(kTrans(keyStr), vTrans(valueStr));
        };
        Translate(filename, std::inserter(map, map.begin()), transformer);
    }

    static bool Useful(const std::string& line) {
        return !line.empty() && (line.front() != '#');
    }

    template <typename TOutputIterator, typename TTransformer>
    static void Translate(const std::string& file, TOutputIterator out, TTransformer trans) {
        std::ifstream ifs(file, std::ios_base::in);
        if (!ifs) {
            throw std::runtime_error("failed to open file " + file);
        }
        std::string line;
        while (std::getline(ifs, line)) {
            boost::trim(line);
            if (!Useful(line)) {
                continue;
            }
            *out = trans(line);
        }
    }

    template <typename TOutputIterator>
    static void Copy(const std::string& file, TOutputIterator out) {
        Translate(file, out, NConverters::TEquivalent{});
    }
};

}   // namespace NNwSmtp
