#include "client.h"
#include "errors.h"

#include <mail/nwsmtp/src/log.h>

#include <yplatform/yield.h>
#include <yplatform/log.h>

#include <sstream>

namespace NFouras {

static const std::string FOURAS{"FOURAS"};

TClient::TClient(
    THttpClientPtr httpClient,
    std::string domain,
    NNwSmtp::TContextPtr context,
    TCallback callback
)
    : HttpClient(std::move(httpClient))
    , Domain(std::move(domain))
    , Context(std::move(context))
    , Callback(std::move(callback))
{}

void TClient::operator()(TYieldCtx yieldCtx, boost::system::error_code ec, yhttp::response response) {
    TKeyEntry key;

    reenter (yieldCtx) {
        StartedAt = yplatform::time_traits::clock::now();

        yield HttpClient->async_run(
            Context->CreateTaskContext(FOURAS + ":"),
            yhttp::request::GET("/smtp/key" + yhttp::url_encode({{"domain", Domain}})),
            yieldCtx);

        if (!ec) {
            try {
                std::tie(ec, key) = ParseResponse(response);
            } catch (const std::exception& exp) {
                ec = EC_PARSE_ERROR;
                NWLOG_L_EXC(error, FOURAS, "fail to parse response, http_code=" + std::to_string(
                    response.status), exp);
            }
        }

        WriteLog(ec);
        Callback(std::move(ec), std::move(key));
    }
}

void TClient::WriteLog(const boost::system::error_code& ec) {
    std::string description = ec.message();
    bool isError = false;

    if (!ec) {
        description = "Found";
    } else if (ec != EC_DOMAIN_NOT_FOUND && ec != EC_DISABLED) {
        isError = true;
    }

    std::stringstream ss;
    ss << "request domain=" << Domain
       << ", delay=" << yplatform::time_traits::to_string(yplatform::time_traits::clock::now() - StartedAt)
       << ", status=" << (isError ? "error" : "ok")
       << ", reason='" << description << "'";

    if (isError) {
        NWLOG_L(error, FOURAS, ss.str());
    } else {
        NWLOG_L(notice, FOURAS, ss.str());
    }
}

} // namespace NFouras
