#include "parser.h"

#include <butil/network/idn.h>

#include <boost/property_tree/json_parser.hpp>

#include <sstream>

namespace NFouras {

std::pair<EErrorCode, TKeyEntry> ParseResponse(const yhttp::response& response) {
    if (response.status == 200) {
        return ParseSuccessResponse(response.body);
    } else if (response.status == 404) {
        return {ParseNotFoundResponse(response.body), {}};
    } else {
        return {FromHttpStatus(response.status), {}};
    }
}

std::pair<EErrorCode, TKeyEntry> ParseSuccessResponse(const std::string& body) {
    TKeyEntry key;
    std::stringstream ss(body);
    boost::property_tree::ptree tree;
    boost::property_tree::read_json(ss, tree);

    auto status = tree.get<std::string>("status");
    if (status != "ok") {
        throw std::runtime_error("unknown status of response, expected 'ok', but '" + status + "'");
    }
    const auto& respPt = tree.get_child("response");
    auto enabled = respPt.get<bool>("enabled");
    if (!enabled) {
        return {EC_DISABLED, std::move(key)};
    }

    key.Domain = idna::encode(respPt.get<std::string>("domain"));
    key.Selector = idna::encode(respPt.get<std::string>("selector"));
    key.SecretKey = respPt.get<std::string>("private_key");

    if (key.Domain.empty()) {
        throw std::runtime_error("empty domain");
    }
    if (key.Selector.empty()) {
        throw std::runtime_error("empty selector");
    }
    if (key.SecretKey.empty()) {
        throw std::runtime_error("empty secret key");
    }

    return {EC_OK, std::move(key)};
}

EErrorCode ParseNotFoundResponse(const std::string& body) {
    std::stringstream ss(body);
    boost::property_tree::ptree tree;
    boost::property_tree::read_json(ss, tree);

    auto status = tree.get<std::string>("status");
    if (status != "error") {
        throw std::runtime_error("unknown status of response, expected 'error', but '" + status + "'");
    }
    const auto& respPt = tree.get_child("response");
    std::string code = respPt.get<std::string>("code");
    if (code != "not_found") {
        throw std::runtime_error("unknown code, expected 'not_found', but '" + code + "'");
    }
    std::string message = respPt.get<std::string>("message");

    return (message == "Object not found") ? EC_DOMAIN_NOT_FOUND : EC_NOT_FOUND;
}

} // namespace NFouras
