#pragma once

#include <boost/range/iterator_range.hpp>
#include <boost/function.hpp>

/**
 * @brief Find header names and values in range [begin, end).
 * @note Syntax of message is described in rfc822/2822/5322.
 *      This algorithm differs from RFC in the following:
 *          It handles LF as delimiter as well as CRLF.
 *          @mafanasev: May be because of incoming messages from internet are often consist of
 *          LF as delimiter. We also do so in local delivery agent (currently notsolitesrv).
 *          Outgoing messages from NwSmtp and Notsolitesrv are composed using CRLF to be compliant with RFC.
 * @see https://tools.ietf.org/html/rfc5322#section-3.5
 * @returns Pair of ends_with_header flag and iterator to begin of body.
 */
template <class TRange, class TCallback = std::function<void(TRange, TRange, TRange)>>
std::pair<bool, typename TRange::iterator> parse_header(TRange header, TCallback callback) {
    auto beg = header.begin();
    const auto end = header.end();
    bool ends_with_header = false;
    while (beg != end)
    {
        char c = *beg;
        // Check for end of headers (empty line): although RFC-822 recommends
        // to use CRLF for header/body separator (see 4.1 SYNTAX), here, we
        // also check for LF just in case...
        if (c == '\n')
        {
            ++beg;
            break;
        }
        else if (c == '\r' && beg + 1 != end && *(beg + 1) == '\n')
        {
            beg += 2;
            break;
        }
        ends_with_header = false;
        // This line may be a field description
        if (!isspace(c))
        {
            const auto nameStart = beg;  // remember the start position of the line
            while (beg != end && (*beg != ':' && !isspace(*beg)))
                ++beg;
            const auto nameEnd = beg;
            while (beg != end && isspace(*beg))
                ++beg;
            if (beg == end)
                break;
            if (*beg != ':')
            {
                // Humm...does not seem to be a valid header line.
                // Skip this error and advance to the next line
                beg = nameStart;
                while (beg != end && *beg++ != '\n')
                    ;
            }
            else
            {
                // Extract the field name
                const TRange name(nameStart, nameEnd);
                // Skip ':' character
                ++beg;
                // Skip spaces between ':' and the field contents
                while (beg != end && (*beg == ' ' || *beg == '\t'))
                    ++beg;
                const auto ctsStart = beg;
                auto ctsEnd = beg;
                while (beg != end)
                {
                    ctsEnd = beg;
                    while (beg != end)
                    {
                        c = *beg;
                        // Check for end of line
                        if (c == '\r' && beg + 1 != end && *(beg + 1) == '\n')
                        {
                            ctsEnd = beg;
                            beg += 2;
                            ends_with_header = true;
                            break;
                        }
                        else if (c == '\n')
                        {
                            ctsEnd = beg;
                            ++beg;
                            ends_with_header = true;
                            break;
                        }
                        ++beg;
                    }
                    if (beg == end)
                        break;
                    c = *beg;
                    // Handle the case of folded lines
                    if (c == ' ' || c == '\t')
                    {
                        // This is a folding white-space: we keep it as is and
                        // we continue with contents parsing...
                    }
                    else
                    {
                        // End of this field
                        break;
                    }
                    // Check for end of contents
                    if (c == '\r' && beg + 1 != end && *(beg + 1) == '\n')
                    {
                        beg += 2;
                        break;
                    }
                    else if (c == '\n')
                    {
                        ++beg;
                        break;
                    }
                }
                const TRange val(ctsStart, ctsEnd);
                const TRange h(nameStart, ctsEnd);
                callback(name, h, val);
            }
        }
        else
        {
            // Skip this error and advance to the next line
            while (beg != end && *beg++ != '\n')
                ;
        }
    }
    return {ends_with_header, beg};
}
