#pragma once

#include "header_parser.h"

#include <algorithm>
#include <cctype>
#include <iterator>
#include <unordered_map>
#include <unordered_set>
#include <vector>

namespace NNwSmtp {

template <class TRange>
class THeaderStorageIterator;

template <class TRange>
struct THeader {
    THeader(
        TRange name,
        TRange value,
        unsigned int orderNum
    )
        : Name(name)
        , Value(value)
        , OrderNum(orderNum)
    {}
    TRange GetWhole() const {
        return {Name.begin(), Value.end()};
    }
    const TRange Name;
    const TRange Value;
    const unsigned int OrderNum;
};

template <class TRange>
class THeaderStorage {
public:
    using THeaders = std::vector<THeader<TRange>>;
    using TMappedHeaders = std::vector<typename THeaders::size_type>;
    using THeaderMap = std::unordered_map<std::string, TMappedHeaders>;
    using TUniqueHeaderNames = std::unordered_set<std::string>;

    using TAllHeadersRange = boost::iterator_range<typename THeaders::const_iterator>;
    using THeaderStorageRange = boost::iterator_range<THeaderStorageIterator<TRange>>;

    THeaderStorage() = default;

    void Add(TRange name, TRange value);
    template<typename Predicate>
    THeaders FilterHeaders(const Predicate&) const;
    void Clear();

    typename THeaders::size_type CountAll() const;
    typename TMappedHeaders::size_type Count(const std::string& name) const;

    bool Contains(const std::string& name) const;
    size_t Size() const;
    bool Empty() const;

    TAllHeadersRange GetAllHeaders() const;
    THeaderStorageRange GetHeaders(const std::string& name) const;

    const TUniqueHeaderNames& GetUniqueHeaderNames() const;

private:
    template <typename TIterator>
    std::string PrepareKey(TIterator begin, TIterator end) const {
        std::string res;
        res.reserve(std::distance(begin, end));

        std::transform(begin, end, std::back_inserter(res), ::tolower);

        return res;
    }

private:
    THeaders Headers;
    THeaderMap HeaderMap;
    TUniqueHeaderNames UniqueHeaderNames;
};

template <class TRange>
class THeaderStorageIterator {
public:
    using iterator_category = std::bidirectional_iterator_tag;
    using value_type = THeader<TRange>;
    using difference_type = std::ptrdiff_t;
    using pointer = const value_type*;
    using reference = const value_type&;

    THeaderStorageIterator() = default;
    ~THeaderStorageIterator() = default;

    bool operator==(const THeaderStorageIterator& other) const;
    bool operator!=(const THeaderStorageIterator& other) const;

    reference operator*() const;
    pointer operator->() const;

    THeaderStorageIterator& operator++();
    THeaderStorageIterator operator++(int);

    THeaderStorageIterator& operator--();
    THeaderStorageIterator operator--(int);

    friend class THeaderStorage<TRange>;

private:
    using THeaders = typename THeaderStorage<TRange>::THeaders;
    using THeaderMapIterator = typename THeaderStorage<TRange>::THeaderMap::const_iterator;
    using THeaderIterator = typename THeaderStorage<TRange>::TMappedHeaders::const_iterator;

    THeaderStorageIterator(
        const THeaders* headers,
        THeaderMapIterator mapIterator,
        THeaderIterator iterator
    );

private:
    const THeaders* Headers = nullptr;
    THeaderMapIterator MapIterator;
    THeaderIterator Iterator;
};

template <class TRange>
void THeaderStorage<TRange>::Add(TRange name, TRange value) {
    std::string key = PrepareKey(name.begin(), name.end());
    Headers.emplace_back(name, value, Headers.size());
    HeaderMap[key].push_back(Headers.size() - 1);
    UniqueHeaderNames.insert(std::move(key));
}

template<typename TRange>
template<typename Predicate>
typename THeaderStorage<TRange>::THeaders THeaderStorage<TRange>::FilterHeaders(const Predicate& pred) const {
    THeaderStorage<TRange>::THeaders result;
    for (auto&& header : Headers) {
        const std::string key = PrepareKey(header.Name.begin(), header.Name.end());
        if (pred(key)) {
            result.emplace_back(header);
        }
    }
    return result;
}

template <class TRange>
void THeaderStorage<TRange>::Clear() {
    UniqueHeaderNames.clear();
    HeaderMap.clear();
    Headers.clear();
}

template <class TRange>
typename THeaderStorage<TRange>::THeaders::size_type THeaderStorage<TRange>::CountAll() const {
    return Headers.size();
}

template <class TRange>
typename THeaderStorage<TRange>::TMappedHeaders::size_type THeaderStorage<TRange>::Count(const std::string& name) const {
    auto it = HeaderMap.find(PrepareKey(name.begin(), name.end()));
    return it == HeaderMap.end() ? 0 : it->second.size();
}

template <class TRange>
bool THeaderStorage<TRange>::Contains(const std::string& name) const {
    return Count(name) > 0;
}

template <class TRange>
size_t THeaderStorage<TRange>::Size() const {
    return Headers.size();
}

template <class TRange>
bool THeaderStorage<TRange>::Empty() const {
    return Size() == 0;
}

template <class TRange>
typename THeaderStorage<TRange>::TAllHeadersRange THeaderStorage<TRange>::GetAllHeaders() const {
    return {Headers.cbegin(), Headers.cend()};
}

template <class TRange>
typename THeaderStorage<TRange>::THeaderStorageRange THeaderStorage<TRange>::GetHeaders(const std::string& name) const {
    auto it = HeaderMap.find(PrepareKey(name.begin(), name.end()));
    if (it == HeaderMap.end()) {
        return {};
    }
    return {THeaderStorageIterator<TRange>(&Headers, it, it->second.cbegin()), THeaderStorageIterator<TRange>()};
}

template <class TRange>
const typename THeaderStorage<TRange>::TUniqueHeaderNames& THeaderStorage<TRange>::GetUniqueHeaderNames() const {
    return UniqueHeaderNames;
}

template <class TRange>
THeaderStorageIterator<TRange>::THeaderStorageIterator(
    const THeaders* headers,
    THeaderMapIterator mapIterator,
    THeaderIterator iterator
)
    : Headers(headers)
    , MapIterator(mapIterator)
    , Iterator(iterator)
{}

template <class TRange>
bool THeaderStorageIterator<TRange>::operator==(const THeaderStorageIterator<TRange> &other) const {
    return (!Headers && (!other.Headers || other.Iterator == other.MapIterator->second.end())) ||
        (!other.Headers && Iterator == MapIterator->second.end()) ||
        Iterator == other.Iterator;
}

template <class TRange>
bool THeaderStorageIterator<TRange>::operator!=(const THeaderStorageIterator<TRange> &other) const {
    return !operator==(other);
}

template <class TRange>
typename THeaderStorageIterator<TRange>::reference THeaderStorageIterator<TRange>::operator*() const {
    return (*Headers)[*Iterator];
}

template <class TRange>
typename THeaderStorageIterator<TRange>::pointer THeaderStorageIterator<TRange>::operator->() const {
    return Headers->data() + *Iterator;
}

template <class TRange>
THeaderStorageIterator<TRange>& THeaderStorageIterator<TRange>::operator++() {
    ++Iterator;
    return *this;
}

template <class TRange>
THeaderStorageIterator<TRange> THeaderStorageIterator<TRange>::operator++(int) {
    auto old = *this;
    ++(*this);
    return old;
}

template <class TRange>
THeaderStorageIterator<TRange>& THeaderStorageIterator<TRange>::operator--() {
    --Iterator;
    return *this;
}

template <class TRange>
THeaderStorageIterator<TRange> THeaderStorageIterator<TRange>::operator--(int) {
    auto old = *this;
    --(*this);
    return old;
}

template <class TIterator, class TRange = boost::iterator_range<TIterator>>
auto ParseMessage(TIterator begin, TIterator end) {
    using THeaders = THeaderStorage<TRange>;
    THeaders headers{};
    auto [ok, iterBody] = parse_header<TRange>({begin, end},
        [&headers](const auto& name, const auto&, const auto& value) {
            headers.Add(name, value);
        });
    if (!ok) {
        return std::pair<THeaders, TRange>{};
    }
    return std::pair<THeaders, TRange>{headers, {iterBody, end}};
}

template <class TBuffer>
inline auto ParseMessage(const TBuffer& msg) {
    return ParseMessage(msg.cbegin(), msg.cend());
}

}   // namespace NNwSmtp
