#pragma once

#include "options.h"

#include <logdog/logger.h>
#include <logdog/format/tskv.h>
#include <logdog/backend/yplatform_log.h>
#include <logdog/attributes/mail_attributes.h>

#include <yplatform/log.h>

namespace NNwSmtp {

namespace NLog {

LOGDOG_DEFINE_ATTRIBUTE(std::string, connection_id)
LOGDOG_DEFINE_ATTRIBUTE(std::string, envelope_id)
LOGDOG_DEFINE_ATTRIBUTE(std::string, uid)
LOGDOG_DEFINE_ATTRIBUTE(std::string, uniq_id)
LOGDOG_DEFINE_ATTRIBUTE(std::string, host)
LOGDOG_DEFINE_ATTRIBUTE(std::string, smtp_session)
LOGDOG_DEFINE_ATTRIBUTE(std::string, from)
LOGDOG_DEFINE_ATTRIBUTE(std::string, from_uid)
LOGDOG_DEFINE_ATTRIBUTE(std::string, remote_ip)
LOGDOG_DEFINE_ATTRIBUTE(std::string, rcpt)
LOGDOG_DEFINE_ATTRIBUTE(std::string, rcpt_uid)
LOGDOG_DEFINE_ATTRIBUTE(std::uint32_t, inactive_for)
LOGDOG_DEFINE_ATTRIBUTE(std::string, message_id)
LOGDOG_DEFINE_ATTRIBUTE(std::string, client)
LOGDOG_DEFINE_ATTRIBUTE(std::string, x_request_id)
LOGDOG_DEFINE_ATTRIBUTE(std::string, so_status)
LOGDOG_DEFINE_ATTRIBUTE(std::string, so_labels)
LOGDOG_DEFINE_ATTRIBUTE(std::string, avir_status)
LOGDOG_DEFINE_ATTRIBUTE(std::string, sent_status)
LOGDOG_DEFINE_ATTRIBUTE(std::string, remote_answer)
LOGDOG_DEFINE_ATTRIBUTE(std::string, cluster_name)
LOGDOG_DEFINE_ATTRIBUTE(bool, is_corp)
LOGDOG_DEFINE_ATTRIBUTE(std::string, domain_type)
LOGDOG_DEFINE_ATTRIBUTE(std::string, notify_mode)

using logdog::attr::exception;
using logdog::attr::message;
using logdog::attr::where_name;
using logdog::error_code;

constexpr static auto TSKV_FORMATTER{::logdog::tskv::make_formatter(
        BOOST_HANA_STRING("mail-nwsmtp-tskv-log"))};
constexpr static auto TSKV_ANALYTICS_FORMATTER{::logdog::tskv::make_formatter(
        BOOST_HANA_STRING("mail-nwsmtp-tskv_analytics-log"))};

} // namespace NLog

static inline auto CreateTskvLogger(std::string logName) {
    return logdog::make_log(NLog::TSKV_FORMATTER, std::make_shared<yplatform::log::source>(
            YGLOBAL_LOG_SERVICE, logName));
}

static inline auto CreateTskvAnalyticsLogger(std::string logName) {
    return logdog::make_log(NLog::TSKV_ANALYTICS_FORMATTER, std::make_shared<yplatform::log::source>(
            YGLOBAL_LOG_SERVICE, logName));
}

using TTskvLogger = decltype(CreateTskvLogger(std::string{}));
using TTskvAnalyticsLogger = decltype(CreateTskvAnalyticsLogger(std::string{}));

struct TLog {
    TLog();
    yplatform::log::source yplatform;
    TTskvLogger tskv;
    TTskvAnalyticsLogger tskvAnalytics;
};

extern std::shared_ptr<TLog> glog;

} // namespace NNwSmtp

#define NWLOG(level, clusterName, hostName, ...) LOGDOG_(NNwSmtp::glog->tskv, level, \
    NNwSmtp::NLog::cluster_name=clusterName, \
    NNwSmtp::NLog::host=hostName, \
    __VA_ARGS__)

#define NWLOG_MSG(level, msg) NWLOG(level, ::NNwSmtp::gconfig->clusterName, ::NNwSmtp::gconfig->hostName, \
    NNwSmtp::NLog::message=(msg))

#define NWLOG_L(level, where, msg) NWLOG(level, Context->GetClusterName(), Context->GetHostName(), \
    NNwSmtp::NLog::connection_id=Context->GetConnectionId(), \
    NNwSmtp::NLog::envelope_id=Context->GetEnvelopeId(), \
    NNwSmtp::NLog::uniq_id=Context->GetSessionId(), \
    NNwSmtp::NLog::where_name=(where), \
    NNwSmtp::NLog::message=(msg))

#define NWLOG_L_EXC(level, where, msg, exc) NWLOG(level, Context->GetClusterName(), Context->GetHostName(), \
    NNwSmtp::NLog::connection_id=Context->GetConnectionId(), \
    NNwSmtp::NLog::envelope_id=Context->GetEnvelopeId(), \
    NNwSmtp::NLog::uniq_id=Context->GetSessionId(), \
    NNwSmtp::NLog::where_name=(where), \
    NNwSmtp::NLog::message=(msg), \
    NNwSmtp::NLog::exception=(exc))

#define NWLOG_CTX(level, ctx, where, msg) NWLOG(level, (ctx)->GetClusterName(), (ctx)->GetHostName(), \
    NNwSmtp::NLog::connection_id=(ctx)->GetConnectionId(), \
    NNwSmtp::NLog::envelope_id=(ctx)->GetEnvelopeId(), \
    NNwSmtp::NLog::uniq_id=(ctx)->GetSessionId(), \
    NNwSmtp::NLog::where_name=(where), \
    NNwSmtp::NLog::message=(msg))

#define NWLOG_L_EC(level, where, msg, ec) NWLOG(level, Context->GetClusterName(), Context->GetHostName(), \
    NNwSmtp::NLog::connection_id=Context->GetConnectionId(), \
    NNwSmtp::NLog::envelope_id=Context->GetEnvelopeId(), \
    NNwSmtp::NLog::uniq_id=Context->GetSessionId(), \
    NNwSmtp::NLog::where_name=(where), \
    NNwSmtp::NLog::message=(msg), \
    NNwSmtp::NLog::error_code=(ec))

#define NWLOG_EC_CTX(level, ctx, where, msg, ec) NWLOG(level, (ctx)->GetClusterName(), (ctx)->GetHostName(), \
    NNwSmtp::NLog::connection_id=(ctx)->GetConnectionId(), \
    NNwSmtp::NLog::envelope_id=(ctx)->GetEnvelopeId(), \
    NNwSmtp::NLog::uniq_id=(ctx)->GetSessionId(), \
    NNwSmtp::NLog::where_name=(where), \
    NNwSmtp::NLog::message=(msg), \
    NNwSmtp::NLog::error_code=(ec))
