#include "command_impl.h"
#include "convert_ec_to_smtp_answer.h"

#include <mail/nwsmtp/src/log.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>
#include <boost/lexical_cast.hpp>

#include <format>
#include <functional>

namespace NNwSmtp::NMailFromCommand {

namespace ph = std::placeholders;

namespace {

const std::string WHERE{"MAILFROM-COMMAND"};

static constexpr const size_t MAIL_ADDR_MAX_LEN = 256;

}

TErrorCode InitialChecks(const TRequest& request, const TConfig& config) {
    if (config.UseAuth && !request.IsAuthenticated) {
        return EError::NotAuthorized;
    }

    if (request.MailAddr.length() > MAIL_ADDR_MAX_LEN) {
        return EError::MailAddrIsTooLong;
    }

    if (!address_valid(request.MailAddr, config.CheckSenderSyntax)) {
        return EError::MailAddrIsInvalid;
    }

    if (config.MessageSizeLimit > 0) {
        auto it = request.Params.find("size");
        if (it != request.Params.end()) {
            std::uint64_t msize = 0;
            if (!boost::conversion::try_lexical_convert(it->second, msize)) {
                return EError::MessageSizeIsInvalid;
            }
            if (msize > config.MessageSizeLimit) {
                NWLOG_MSG(notice, "message size=" + std::to_string(msize));
                return EError::MessageSizeExceedsLimit;
            }
        }
    }

    return EError::Ok;
}

void TMailFromCommandImpl::Run(TErrorCode ec) {
    reenter (RunCoro) {
        InitialChecksErrorCode = InitialChecks(Request, Config);
        if (InitialChecksErrorCode) {
            NWLOG_L_EC(notice, WHERE, "Error occurred on initial checks", InitialChecksErrorCode);
            ResponseErrorCode = InitialChecksErrorCode;
            yield break;
        }

        if (Request.IsAuthenticated || Config.BlackBoxCheckSender) {
            yield RunBlackBoxCoro(EError::Ok);

            if (ec) {
                ResponseErrorCode = std::move(ec);
                yield break;
            }
        }
        if (Config.CheckSPF) {
            yield RunSPFCheck();
        }

        NWLOG_L(notice, "RECV", "from=" + Request.MailAddr);

        yield RunRateSrvCheck();

        if (ec) {
            ResponseErrorCode = std::move(ec);
            yield break;
        }
    }

    if (RunCoro.is_complete()) {
        if (ResponseErrorCode) {
            NWLOG_L_EC(notice, WHERE, "Error occurred on mail from command", ResponseErrorCode);
        }
        Response.SmtpAnswer = ConvertErrorCodeToSmtpAnswer(ResponseErrorCode, Request.MailAddr);
        boost::asio::post(IoContext,
            std::bind(std::move(Callback), std::move(ResponseErrorCode), std::move(Response))
        );
    }
}

void TMailFromCommandImpl::RunBlackBoxCoro(TErrorCode ec) {
    reenter(BlackBoxCoro) {
        BBSessionSuid = Request.BBSessionSuid;

        if (Request.NeedCheckAuth) {

            yield CheckAuth();

            if (ec) {
                BlackBoxCoroErrorCode = ec;
                NWLOG_L_EC(notice, WHERE, "error occurred on check auth", ec);
                yield break;
            }
        }

        yield CheckBb();

        if (ec) {
            BlackBoxCoroErrorCode = ec;
            NWLOG_L_EC(notice, WHERE, "error occurred on check blackbox", ec);
            yield break;
        }

        BlackBoxCoroErrorCode = EError::Ok;
    }

    if (BlackBoxCoro.is_complete()) {
        boost::asio::post(IoContext, std::bind(&TMailFromCommandImpl::Run, shared_from_this(), BlackBoxCoroErrorCode));
    }
};

void TMailFromCommandImpl::CheckAuth() {
    SettingsAuthorizationClient->Run(Context, {Request.Auth}, std::bind(&TMailFromCommandImpl::HandleCheckAuth, shared_from_this(), ph::_1, ph::_2));
}

void TMailFromCommandImpl::HandleCheckAuth(TErrorCode ec, NBlackBox::TResponse bbResponse) {
    auto chkResult = GetAuthResult(ec, bbResponse);
    auto checkStatus = chkResult.StatusCheck;
    auto errorCode = make_error_code(EError::Ok);
    if (checkStatus == check::CHK_TEMPFAIL
        || checkStatus == check::CHK_REJECT
        || checkStatus == check::CHK_DISCARD)
    {
        errorCode = make_error_code(EError::AuthReject);
    } else if (checkStatus == check::CHK_ACCEPT) {
        BBSessionSuid = bbResponse.Suid;
        Response.CheckAuthResponse = TCheckAuthResponse{
            .BBSessionSuid = bbResponse.Suid,
            .BBSessionLogin = bbResponse.Login
        };
    } else {
        errorCode = make_error_code(EError::Unknown);
        Response.CheckAuthResponse = TCheckAuthResponse{};
    }

    boost::asio::post(IoContext, std::bind(&TMailFromCommandImpl::RunBlackBoxCoro, shared_from_this(), std::move(errorCode)));
}

void TMailFromCommandImpl::CheckBb() {
    BbChecks->CheckMailFrom(
        Context,
        Request.MailAddr,
        Request.Ip.to_string(),
        std::bind(&TMailFromCommandImpl::HandleCheckBB, shared_from_this(), ph::_1, ph::_2)
    );
}

void TMailFromCommandImpl::HandleCheckBB(TErrorCode ec, NBlackBox::TResponse bbResponse) {
    auto checkResult = NBlackBox::GetMailfromResult(ec, bbResponse);
    checkResult.BlackBoxResult.AuthAddr = Request.MailAddr;

    NWLOG_L(notice, "BB-MAILFROM", "mailfrom='" + Request.MailAddr + "', status='" + checkResult.LogMsg + "'");
    NWLOG_L(notice, "BB-MAILFROM", "org_id=" + (bbResponse.OrgId.empty() ? "missing" : bbResponse.OrgId));

    auto statusCheck = checkResult.StatusCheck;

    const bool bbSessionSuidDiffers = BBSessionSuid != checkResult.BlackBoxResult.Suid || BBSessionSuid == 0;
    if (Request.IsAuthenticated && bbSessionSuidDiffers) {
        statusCheck = check::CHK_REJECT;
    }

    auto errorCode = make_error_code(EError::Ok);
    if (statusCheck == check::CHK_DISCARD || statusCheck == check::CHK_REJECT) {
        errorCode = make_error_code(Request.IsAuthenticated ? EError::RejectAddrNotOwnedByAuthUser : EError::RejectAddrUserNotFound);
    } else if (statusCheck == check::CHK_TEMPFAIL) {
        errorCode = make_error_code(EError::TempFail);
    }

    if (!errorCode && !checkResult.BlackBoxResult.Uid.empty()) {
        Response.CheckBBResponse = TCheckBBResponse {
            .BBResult = checkResult.BlackBoxResult
        };
    }

    boost::asio::post(IoContext, std::bind(&TMailFromCommandImpl::RunBlackBoxCoro, shared_from_this(), std::move(errorCode)));
}

void TMailFromCommandImpl::RunSPFCheck() {
    SPFCheckClient->Check(
        Request.MailAddr,
        Request.HeloHost,
        Request.Ip,
        [context = Context](const auto& message) {
            NWLOG_CTX(notice, context, "SPF", message);
        },
        std::bind(&TMailFromCommandImpl::HandleSPFCheck, shared_from_this(), ph::_1, ph::_2)
    );
}

void TMailFromCommandImpl::HandleSPFCheck(TErrorCode ec, NSPF::TResponse response) {
    if (ec) {
        NWLOG_L_EC(notice, WHERE, "Error occurred on check spf", ec);
    }

    Response.SPFResponse = TSPFResponse {
        .Result = std::move(response.Result),
        .Explain = std::move(response.Expl)
    };

    boost::asio::post(IoContext, std::bind(&TMailFromCommandImpl::Run, shared_from_this(), EError::Ok));
}

void TMailFromCommandImpl::RunRateSrvCheck() {
    std::string sender = Request.MailAddr;
    std::string senderDefaultEmail = Request.MailAddr;
    if (Response.CheckBBResponse) {
        sender = Response.CheckBBResponse->BBResult.Uid;
        senderDefaultEmail = Response.CheckBBResponse->BBResult.DefaultEmail;
    }
    NRateSrv::TRequest req {
        .Sender = std::move(sender),
        .SenderDefaultEmail = std::move(senderDefaultEmail)
    };
    AsyncCheckSender->Run(Context, std::move(req), std::bind(&TMailFromCommandImpl::HandleRateSrvCheck, shared_from_this(), ph::_1));
}

void TMailFromCommandImpl::HandleRateSrvCheck(TErrorCode ec) {
    TErrorCode responseErrorCode = EError::Ok;
    if (ec == make_error_condition(EError::RejectRateSrv)) {
        NWLOG_L(notice, "RECV", std::format("reject from={} by RateSrv", Request.MailAddr));
        responseErrorCode = EError::RejectRateSrv;
    }
    boost::asio::post(IoContext, std::bind(&TMailFromCommandImpl::Run, shared_from_this(), std::move(responseErrorCode)));
}

#include <boost/asio/unyield.hpp>

TConfig MakeConfig(std::shared_ptr<NNwSmtp::Options> config) {
    return NMailFromCommand::TConfig {
        .BlackBoxCheckSender = config->blackbox.checkSender,
        .CheckSPF = config->spf.use,
        .UseAuth = config->auth.use,
        .CheckSenderSyntax = config->smtpOpts.constraints.checkSenderSyntax,
        .MessageSizeLimit = config->smtpOpts.constraints.messageSizeLimit,
    };
}

} // namespace NNwSmtp::NMailFromCommand
