#pragma once

#include "command.h"
#include "error_code.h"

#include <mail/nwsmtp/src/blackbox/bb_checks.h>
#include <mail/nwsmtp/src/error_code.h>
#include <mail/nwsmtp/src/options.h>
#include <mail/nwsmtp/src/ratesrv/checks.h>
#include <mail/nwsmtp/src/settings_authorization/settings_authorization_impl.h>
#include <mail/nwsmtp/src/spf/spf_check.h>

#include <boost/asio.hpp>

#include <memory>

namespace NNwSmtp::NMailFromCommand {

TErrorCode InitialChecks(const TRequest& request, const TConfig& config);

class TMailFromCommandImpl : public std::enable_shared_from_this<TMailFromCommandImpl> {
public:
    TMailFromCommandImpl(TContextPtr context, TRequest request, TCallback callback,
        TConfig config,
        NSPF::TSPFCheckClientPtr spfCheckClient,
        NBlackBox::TBBChecksPtr bbChecks,
        NSettingsAuthorization::TSettingsAuthorizationPtr settingsAuthorizationClient,
        NRateSrv::TAsyncCheckSenderClientPtr asyncCheckSender,
        boost::asio::io_context& ioContext)
        : Context(std::move(context))
        , Request(std::move(request))
        , Callback(std::move(callback))
        , Config(std::move(config))
        , IoContext(ioContext)
        , SPFCheckClient(std::move(spfCheckClient))
        , BbChecks(std::move(bbChecks))
        , SettingsAuthorizationClient(std::move(settingsAuthorizationClient))
        , AsyncCheckSender(std::move(asyncCheckSender))
    {}

    void Run(TErrorCode ec);

private:
    void RunBlackBoxCoro(TErrorCode ec);

    void CheckAuth();
    void HandleCheckAuth(TErrorCode ec, NBlackBox::TResponse bbResponse);

    void CheckBb();
    void HandleCheckBB(TErrorCode ec, NBlackBox::TResponse bbResponse);

    void RunSPFCheck();
    void HandleSPFCheck(TErrorCode ec, NSPF::TResponse response);

    void RunRateSrvCheck();
    void HandleRateSrvCheck(TErrorCode ec);

private:
    TContextPtr Context;
    TRequest Request;
    TCallback Callback;
    TConfig Config;
    boost::asio::io_context& IoContext;

    NSPF::TSPFCheckClientPtr SPFCheckClient;
    NBlackBox::TBBChecksPtr BbChecks;
    NSettingsAuthorization::TSettingsAuthorizationPtr SettingsAuthorizationClient;
    NRateSrv::TAsyncCheckSenderClientPtr AsyncCheckSender;

    boost::asio::coroutine RunCoro;
    boost::asio::coroutine BlackBoxCoro;

    TErrorCode InitialChecksErrorCode;
    TErrorCode BlackBoxCoroErrorCode;

    NSPF::TResponse SPFResponse;
    std::size_t BBSessionSuid = 0;

    TErrorCode ResponseErrorCode;
    TResponse Response;
};

class TMailFromCommand : public IMailFromCommand {
public:
    TMailFromCommand(TConfig config,
        NSPF::TSPFCheckClientPtr spfCheckClient,
        NBlackBox::TBBChecksPtr bbChecks,
        NSettingsAuthorization::TSettingsAuthorizationPtr settingsAuthorizationClient,
        NRateSrv::TAsyncCheckSenderClientPtr asyncCheckSender,
        boost::asio::io_context& ioContext)
    : Config(std::move(config))
    , SpfCheckClient(std::move(spfCheckClient))
    , BbChecks(std::move(bbChecks))
    , SettingsAuthorizationClient(std::move(settingsAuthorizationClient))
    , AsyncCheckSender(std::move(asyncCheckSender))
    , IoContext(ioContext)
    {}

    void Run(TContextPtr context, TRequest request, TCallback cb) override {
        auto impl = std::make_shared<TMailFromCommandImpl>(std::move(context), std::move(request), std::move(cb),
            Config, SpfCheckClient, BbChecks, SettingsAuthorizationClient, AsyncCheckSender, IoContext);
        boost::asio::post(IoContext, std::bind(&TMailFromCommandImpl::Run, std::move(impl), EError::Ok));
    }

private:
    TConfig Config;
    NSPF::TSPFCheckClientPtr SpfCheckClient;
    NBlackBox::TBBChecksPtr BbChecks;
    NSettingsAuthorization::TSettingsAuthorizationPtr SettingsAuthorizationClient;
    NRateSrv::TAsyncCheckSenderClientPtr AsyncCheckSender;
    boost::asio::io_context& IoContext;
};

TConfig MakeConfig(std::shared_ptr<NNwSmtp::Options> config);

} // namespace NNwSmtp::NMailFromCommand
