#pragma once

#include <mail/nwsmtp/src/ratesrv/error_code.h>

#include <boost/system/error_code.hpp>

namespace NNwSmtp::NMailFromCommand {

enum class EError {
    Ok,
    MailAddrIsTooLong,
    MailAddrIsInvalid,
    MessageSizeIsInvalid,
    MessageSizeExceedsLimit,
    NotAuthorized,
    AuthReject,
    RejectAddrNotOwnedByAuthUser,
    RejectAddrUserNotFound,
    RejectRateSrv,
    TempFail,
    Unknown
};

} // namespace NNwSmtp::NMailFromCommand

namespace boost::system {

template <>
struct is_error_code_enum<NNwSmtp::NMailFromCommand::EError> : std::true_type {};

} // namespace boost::system

namespace NNwSmtp::NMailFromCommand {

class TErrorCategory final : public boost::system::error_category {

public:
    const char* name() const noexcept override {
        return "nwsmtp_mail_from";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::Ok:
                return "Ok";
            case EError::MailAddrIsTooLong:
                return "Mail address is too long";
            case EError::MailAddrIsInvalid:
                return "Mail address is invalid";
            case EError::MessageSizeIsInvalid:
                return "Message size is invalid";
            case EError::MessageSizeExceedsLimit:
                return "Message size exceeds limit";
            case EError::NotAuthorized:
                return "Not authorized";
            case EError::AuthReject:
                return "Authorization reject";
            case EError::RejectAddrNotOwnedByAuthUser:
                return "Reject: address is not owned by authorized user";
            case EError::RejectAddrUserNotFound:
                return "Reject: address user not found";
            case EError::RejectRateSrv:
                return "Rejected by rate limits";
            case EError::TempFail:
                return "Temporary fail";
            case EError::Unknown:
                return "Unknown fail";
        }
        return "Unknown error code: " + std::to_string(value);
    }

    bool equivalent(const boost::system::error_code& code, int condition) const noexcept override {
        switch (EError {condition}) {
            case EError::RejectRateSrv:
                return code == NRateSrv::EError::Reject
                    || code == NRateSrv::EError::Discard;
            default:
                return boost::system::error_category::equivalent(code, condition);
        }
    }
};

inline const TErrorCategory& getErrorCategory() {
    static TErrorCategory errorCategory;
    return errorCategory;
}

inline auto make_error_code(EError ec) {
    return boost::system::error_code(static_cast<int>(ec), getErrorCategory());
}

inline auto make_error_condition(EError ec) {
    return boost::system::error_condition(static_cast<int>(ec), getErrorCategory());
}

} // namespace NNwSmtp::NMailFromCommand
