#include <iostream>
#include <boost/bind.hpp>
#include <boost/format.hpp>
#include <boost/algorithm/string/case_conv.hpp>
#include <boost/algorithm/string/trim.hpp>

#include <yplatform/find.h>

#include <yplatform/coroutine.h>

#include <boost/property_tree/json_parser.hpp>

#include "client_impl.h"

#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/utils.h>

#include <yplatform/yield.h>

class list_client_category : public boost::system::error_category
{
public:
    const char * name() const BOOST_NOEXCEPT override
    {
        return "list client";
    }

    std::string message(int ev) const override
    {
        switch (ev)
        {
        case PARSE_ERROR:
            return "Failed to parse list";
        case NOT_FOUND:
            return "Not found";
        case INVALID_CODE:
            return "Invalid code";
        default:
            return "Unknown list client error";
        }
    }
};

const boost::system::error_category & get_list_client_category()
{
    static list_client_category instance;
    return instance;
}

boost::system::error_code make_error_code(list_client_ec e)
{
    return boost::system::error_code(e, get_list_client_category());
}


namespace NNwSmtp::NML {

static const std::string LIST{"LIST"};

yhttp::request makeRequest(const std::string& baseUrl, const std::string& email) {
    auto url = baseUrl + email;
    return yhttp::request::GET(std::move(url));
}

yhttp::options makeOptions(const Options::LLOpts& llOpts) {
    yhttp::options opts;
    opts.timeouts.connect = llOpts.connectTimeout;
    opts.timeouts.total = llOpts.timeout;
    return opts;
}

inline std::string normalize(const std::string& s) {
    return boost::trim_copy(boost::to_lower_copy(s));
}

TResponse parse_ml_data(const std::string& data) {
    TResponse result;
    std::stringstream ss(data);

    boost::property_tree::ptree tree;
    boost::property_tree::read_json(ss, tree);

    result.Internal = tree.get<bool>("is_internal", false);
    result.Open = tree.get<bool>("is_open", false);
    result.Readonly = tree.get<bool>("readonly", false);

    if (auto inboxOpt = tree.get_child_optional("subscribers.inbox")) {
        for (const auto& val : inboxOpt.get()) {
            result.Subscribers.push_back(normalize(val.second.data()));
        }
    }
    if (auto emailsOpt = tree.get_child_optional("emails")) {
        for (const auto& val : emailsOpt.get()) {
            result.Subscribers.push_back(normalize(val.second.data()));
        }
    }
    if (auto whoCanWriteOpt = tree.get_child_optional("who_can_write")) {
        for (const auto& val : whoCanWriteOpt.get()) {
            result.WhoCanWrite.insert(normalize(val.second.data()));
        }
    }
    return result;
}

struct Impl {
    Impl(const Options::LLOpts& settings, std::shared_ptr<yhttp::simple_call> client)
        : settings(settings)
        , httpClient(client)
    {}

    using YieldContext = yplatform::yield_context<Impl>;

    void operator()(
        YieldContext yieldCtx,
        boost::system::error_code errc = boost::system::error_code(),
        yhttp::response response = yhttp::response())
    {
        if (errc == boost::asio::error::operation_aborted) {
            return;
        }
        reenter(yieldCtx) {
            for (attempt = 0; attempt < settings.attempts; ++attempt) {
                startedAt = yplatform::time_traits::clock::now();

                yield httpClient->async_run(
                    Context->CreateTaskContext(LIST + ":"),
                    makeRequest(settings.addr.str(), data.Email),
                    makeOptions(settings),
                    yieldCtx);

                if (errc) {
                    logError(errc, response);
                    continue;
                }

                if (response.status != 200) {
                    if (response.reason.empty()) {
                        response.reason = std::to_string(response.status);
                    }
                    errc = make_error_code(response.status == 410 ? NOT_FOUND : INVALID_CODE);
                    logError(errc, response);
                    if (response.status / 100 == 5) {
                        continue;
                    } else {
                        yield break;
                    }
                }

                try {
                    mlResponse = parse_ml_data(response.body);
                    logSuccess();
                } catch (const std::exception& e) {
                    errc = make_error_code(PARSE_ERROR);
                    response.reason = e.what();
                    logError(errc, response);
                }
                yield break;
            }   // for
        }   // reenter

        if (yieldCtx.is_complete()) {
            callback(errc, std::move(mlResponse));
        }
    }

    void logError(boost::system::error_code errc, const yhttp::response& resp) const {
        NWLOG_L(error, LIST, "try=" + std::to_string(attempt + 1) + ", email='" + data.Email + "'" +
            ", delay=" + yplatform::time_traits::to_string(yplatform::time_traits::clock::now() - startedAt) +
            ", stat=error (" + errc.message() + (resp.reason.empty() ? "" : (": " + resp.reason)) + ")");
    }

    void logSuccess() const {
        NWLOG_L(notice, LIST, "try=" + std::to_string(attempt + 1) + ", email='" + data.Email + "'" +
            ", delay=" + yplatform::time_traits::to_string(yplatform::time_traits::clock::now() - startedAt) +
            ", stat=ok");
    }

    const Options::LLOpts& settings;
    std::shared_ptr<yhttp::simple_call> httpClient;
    unsigned short attempt = 0;
    TContextPtr Context;
    TClient::TCallback callback;
    TRequest data;
    TResponse mlResponse;
    yplatform::time_traits::time_point startedAt;
};

void TClient::Run(TContextPtr context, TRequest request, TCallback callback) {
    auto impl = std::make_shared<Impl>(Settings, HttpClient);
    impl->Context = std::move(context);
    impl->data = std::move(request);
    impl->callback = std::move(callback);
    yplatform::spawn(Io.get_executor(), impl);
}

}   // namespace NNwSmtp::NML
