#include "client_impl.h"
#include "error_code.h"
#include "make_response.h"
#include "types_reflection.h"

#include <mail/yreflection/include/yamail/data/serialization/json_writer.h>
#include <mail/yreflection/include/yamail/data/deserialization/json_reader.h>

#include <mail/nwsmtp/src/log.h>

#include <functional>

namespace NNwSmtp::NNsls {

namespace {

auto MakeRequest(std::string requestId, std::string body) {
    return yhttp::request::POST("/store?request_id=" +  requestId, std::move(body));
}

} // namespace anonymous

void TNslsClient::Store(TContextPtr context, TRequest request, TCallback callback) {

    std::string body  = yamail::data::serialization::toJson(request).str();

    auto handler = [ex = Io.get_executor(), callback = std::move(callback), context = context]
        (const auto& ec, auto response) mutable {
            boost::asio::post(ex,
                std::bind(&TNslsClient::HandleStore, std::move(context), std::move(callback), ec, std::move(response)));
        };

    Client->async_run(context, MakeRequest(request.requestId, std::move(body)), std::move(handler));
}

void TNslsClient::HandleStore(
    TContextPtr context,
    TCallback callback,
    const TErrorCode& ec,
    yhttp::response httpResponse
) {
    if (ec) {
        return callback(ec, {});
    }

    if (httpResponse.status == 406) {
        return callback(EError::EC_PERM_ERROR, {});
    } else if (httpResponse.status != 200) {
        return callback(EError::EC_BAD_STATUS, {});
    }

    TStoreResponse storeResponse;
    try {
        yamail::data::deserialization::fromJson<TStoreResponse>(httpResponse.body, storeResponse);
    } catch (const std::exception& e) {
        using namespace std::string_literals;
        NWLOG_CTX(error, context, "NSLS", "failed to parse request: "s + e.what());
        return callback(EError::EC_PARSE_ERROR, {});
    }

    callback({}, MakeResponse(storeResponse));
}

} // namespace NNwSmtp::NNsls
