#include "impl.h"

#include "types.h"

#include "interface.h"
#include "util/system/types.h"
#include <mail/nwsmtp/src/options.h>

#include <mail/nwsmtp/src/mailfrom/command.h>
#include <mail/nwsmtp/src/rcpt_to/interface.h>
#include <mail/nwsmtp/src/so/client_impl.h>
#include <mail/nwsmtp/src/delivery/async/interface.h>

#include <mail/nwsmtp/src/settings_authorization/settings_authorization.h>
#include <mail/library/received_header/received_header.h>

#include <mail/nwsmtp/src/context.h>
#include <mail/nwsmtp/src/randgen.h>
#include <mail/nwsmtp/src/types.h>

#include <boost/algorithm/string/case_conv.hpp>

#include <boost/asio/io_context.hpp>
#include <boost/asio/ip/address.hpp>

#include <memory>

namespace NNwSmtp {

namespace ph = std::placeholders;

namespace {

using TParams = std::map<std::string, std::string>;

TParams ParamsKeysToLowerCase(const TParams& params) {
    TParams lowerKeysParams;
    for (const auto& [key, value]: params) {
        lowerKeysParams.emplace(std::make_pair(boost::to_lower_copy(key), value));
    }
    return lowerKeysParams;
}

}

TNwsmtpImpl::TNwsmtpImpl(NMailFromCommand::TMailFromCommandPtr mailFromCmd,
    NRcptTo::TRcptToCommandPtr rcptToCmd,
    NSettingsAuthorization::TSettingsAuthorizationPtr settingsAuthorization,
    NAsyncDlv::TAsyncDeliveryPtr delivery,
    boost::asio::io_context& ioContext
)
    : MailFromCmd(std::move(mailFromCmd))
    , RcptToCmd(std::move(rcptToCmd))
    , SettingsAuthorization(std::move(settingsAuthorization))
    , DeliveryClient(std::move(delivery))
    , SessionId(NNwSmtp::GenRandomString())
    , IoContext(ioContext)
{
    Reset();
}

void TNwsmtpImpl::SetEhloHost(std::string host) {
    EhloHost = std::move(host);
}

void TNwsmtpImpl::SetRemoteIp(boost::asio::ip::address addr) {
    RemoteAddr = std::move(addr);
}

void TNwsmtpImpl::SetRemoteHostName(std::string host) {
    RemoteHostName = std::move(host);
}

void TNwsmtpImpl::Auth(TAuthRequst req, TAuthCallback cb) {
    auto context{boost::make_shared<TContext>(SessionId, std::string{}, gconfig->clusterName, gconfig->hostName)};
    NSettingsAuthorization::TRequest request {
        .Data = req.Auth
    };
    AuthData = req.Auth;
    SettingsAuthorization->Run(std::move(context), std::move(request),
        std::bind(&TNwsmtpImpl::HandleAuth, shared_from_this(), ph::_1, ph::_2, std::move(cb)));
}

bool TNwsmtpImpl::IsAuthenticated() {
    return IsAuthenticatedFlag;
}

void TNwsmtpImpl::MailFrom(TMailFromRequest req, TCallback cb) {
    ++MessagesCount;
    MailFromAddr = req.MailAddr;

    if (auto envIdIt = req.Params.find("envid");
        envIdIt != req.Params.end() && envIdIt->second.length() <= 100) {
        OriginalEnvId = envIdIt->second;
    }

    bool needCheckAuth = IsAuthenticatedFlag &&
        gconfig->smtpOpts.maxMessagesPerAuth &&
        (AuthCheckedOnMessageIdx + gconfig->smtpOpts.maxMessagesPerAuth <= MessagesCount);

    NMailFromCommand::TRequest request {
        .MailAddr = std::move(req.MailAddr),
        .Params = ParamsKeysToLowerCase(req.Params),
        .HeloHost = EhloHost,
        .Ip = RemoteAddr,
        .BBSessionSuid = BBSessionSuid,
        .IsAuthenticated = IsAuthenticatedFlag,
        .NeedCheckAuth = needCheckAuth,
        .Auth = AuthData
    };

    MailFromCmd->Run(Context, std::move(request),
        std::bind(&TNwsmtpImpl::HandleMailFrom, shared_from_this(), ph::_1, ph::_2, std::move(cb))
    );
}

void TNwsmtpImpl::RcptTo(TRcptToRequest req, TRcptToCallback cb) {
    NRcptTo::TRequest request {
        .MailFrom = Envelope->m_sender,
        .RcptTo = std::move(req.MailAddr),
        .RemoteIp = RemoteAddr,
        .Params = ParamsKeysToLowerCase(req.Params),
        .Envelope = Envelope,
    };
    RcptToCmd->Run(Context, std::move(request),
        std::bind(&TNwsmtpImpl::HandleRcptTo, shared_from_this(), ph::_1, ph::_2, std::move(cb))
    );
}

void TNwsmtpImpl::Data(TDataRequest req) {
    NReceived::TReceived received;
    received.Date = NUtil::GetRfc822DateNow();
    if (!gconfig->msgOpts.hideSourceInReceived) {
        received.HeloHost = EhloHost;
        received.RemoteHost = RemoteHostName;
        received.RemoteIp = RemoteAddr.to_string();
    }
    received.LocalHost = boost::asio::ip::host_name();
    received.ClusterName = gconfig->soOpts.ClusterName;
    received.Protocol = req.ProtocolType;
    received.SessionId = Context->GetSessionId();
    received.Ssl = req.Ssl;

    Envelope->received_headers_ = NReceived::BuildReceivedHeader(received);
}

void TNwsmtpImpl::Delivery(TDeliveryRequest req, TDeliveryCallback cb) {
    NWLOG_L(notice, "RECV", "message size=" + std::to_string(req.MsgSize));

    if (Envelope->m_rcpts.Empty()) {
        NWLOG_L(notice, "RECV", "all recipients discarded");
        return boost::asio::post(IoContext, std::bind(std::move(cb), NSmtp::EError::Discarded, "250 2.0.0 Ok, discarded"));
    }

    Envelope->orig_message_ = req.Msg;
    Envelope->orig_message_size_ = req.MsgSize;

    if (Envelope->orig_message_size_ > gconfig->smtpOpts.constraints.messageSizeLimit) {
        ++ErrorCount;

        NWLOG_L(notice, "RECV", "warning: queue file size limit exceeded");
        return boost::asio::post(IoContext, std::bind(std::move(cb), NSmtp::EError::Rejected, "552 5.3.4 Error: message file too big;"));
    }

    if (IsAuthenticatedFlag) {
        using namespace std::string_literals;
        NWLOG_L(notice, "", "login=" + BBSessionLogin);
    }

    if (gconfig->deliveryToSenderControl.use
        && !Envelope->is_sender_in_rcpts_
        && !Envelope->m_sender_uid.empty()
    ) {
        Envelope->add_recipient(
            /*rcpt=                */ Envelope->m_sender,
            /*suid=                */ 0,
            /*is_alias=            */ true,
            /*uid=                 */ Envelope->m_sender_uid,
            /*ml_uid=              */ "",
            /*default_email=       */ "",
            /*country_code=        */ "",
            /*domainType=          */ DomainType::LOCAL,
            /*notify_mode=         */ dsn::Options::NEVER,
            /*phone_confirmed=     */ false,
            /*bool is_maillist=    */ false,
            /*std::time_t regDate= */ 0,
            /*need_update_limits=  */ false
        );
        Envelope->sender_added_for_control = true;
    }

    auto dlvRequest = NAsyncDlv::TRequest {
        EhloHost,
        SessionId,
        MailFromAddr,
        RemoteHostName,
        Envelope,
        RemoteAddr,
        false,
        SpfResponse,
        SpfExplain,
        time(NULL)
    };

    DeliveryClient->Run(
        Context,
        std::move(dlvRequest),
        std::bind(&TNwsmtpImpl::HandleDeliver, shared_from_this(), ph::_1, ph::_2, std::move(cb))
    );
}

void TNwsmtpImpl::Reset() {
    Envelope = boost::make_shared<envelope>();
    Context = boost::make_shared<TContext>(SessionId, Envelope->m_id, gconfig->clusterName, gconfig->hostName);
}

TContextPtr TNwsmtpImpl::GetContext() {
    return Context;
}

void TNwsmtpImpl::HandleAuth(TErrorCode ec, NBlackBox::TResponse response, TAuthCallback cb) {
    if (!ec) {
        BBSessionSuid = response.Suid;
        BBSessionLogin = response.Login;
        IsAuthenticatedFlag = true;
    }

    return boost::asio::post(IoContext, std::bind(std::move(cb), std::move(ec), std::move(response)));
}

void TNwsmtpImpl::HandleMailFrom(TErrorCode ec, NMailFromCommand::TResponse resp, TCallback cb) {
    if (ec) {
        return boost::asio::post(IoContext, std::bind(std::move(cb), std::move(ec)));
    }

    if (resp.CheckAuthResponse) {
        AuthCheckedOnMessageIdx = MessagesCount;
        BBSessionSuid = std::move(resp.CheckAuthResponse->BBSessionSuid);
        BBSessionLogin = std::move(resp.CheckAuthResponse->BBSessionLogin);
    }

    if (resp.CheckBBResponse) {
        Envelope->senderInfo = resp.CheckBBResponse->BBResult;
        Envelope->m_sender_uid = std::move(resp.CheckBBResponse->BBResult.Uid);
        Envelope->m_sender_default_email = std::move(resp.CheckBBResponse->BBResult.DefaultEmail);
    } else {
        Envelope->senderInfo = std::nullopt;
    }

    if (resp.SPFResponse) {
        SpfResponse = std::move(resp.SPFResponse->Result);
        SpfExplain = std::move(resp.SPFResponse->Explain);
    }

    if (OriginalEnvId) {
        Envelope->m_original_id = *OriginalEnvId;
    }

    Envelope->m_sender = MailFromAddr.empty() ? "<>" : MailFromAddr;
    return boost::asio::post(IoContext, std::bind(std::move(cb), NMailFromCommand::EError::Ok));
}

void TNwsmtpImpl::HandleRcptTo(TErrorCode ec, NRcptTo::TResponse resp, TRcptToCallback cb) {
    boost::asio::post(IoContext, std::bind(std::move(cb), std::move(ec), std::move(resp.SmtpAnswer)));
}

void TNwsmtpImpl::HandleDeliver(TErrorCode ec, std::string answer, TDeliveryCallback cb) {
    boost::asio::post(IoContext, std::bind(std::move(cb), std::move(ec), std::move(answer)));
}

} // namespace NNwSmtp
