#pragma once

#include <iostream>
#include <regex>
#include <set>
#include <vector>
#include <chrono>
#include <optional>
#include <boost/asio/ip/tcp.hpp>
#include <boost/unordered_set.hpp>
#include <nwsmtp/remote_point.h>
#include <nwsmtp/ares_options.h>
#include <nwsmtp/avir_options.h>
#include <yplatform/ptree.h>
#include <yplatform/reactor.h>

#include <ymod_smtpclient/request.h>
#include <ymod_smtpclient/smtp_point.h>

#include <mail/library/dsn/options.hpp>
#include <mail/nwsmtp/src/so/config.h>

#include "control_from/options.h"
#include "log.h"
#include "match.h"
#include "utils.h"
#include "domains_set.h"
#include "router/settings.h"

namespace NNwSmtp {

using namespace std::literals::chrono_literals;

using ptree = yplatform::ptree;

using SmtpTimeouts = ymod_smtpclient::Timeouts;
using SmtpPoint = ymod_smtpclient::SmtpPoint;

struct Options {
    using Duration = yplatform::time_traits::duration;

    struct SmtpConnectionOpts {
        bool startTls = false;
        // limits
        unsigned int maxMessagesPerAuth = 0;    // 0 - no use

        struct ProtoConstraints {
            bool checkSenderSyntax;
            bool strictAsciiRecipient;
            bool allowPercentHack;
            unsigned int rcptCountLimit;
            std::size_t messageSizeLimit;
        };
        ProtoConstraints constraints;

        std::string banner;

        struct SessionTimeLimit {
            bool use = false;
            Duration timeout = yplatform::time_traits::minutes(30);
        };

        SessionTimeLimit sessionTimeLimit;
        std::size_t unrecognizedCommandsMaxCount = 10;

        bool useRateLimiter = false;

        void init(const ptree& pt);
    };

    struct MessageProcessingOpts {
        boost::unordered_set<std::string> removeHeadersSet;
        unsigned int headersCountLimit;

        bool hideSourceInReceived;
        bool rewriteSenderFromHeader;
        bool addReturnPath;
        bool markMLMessage;

        NControlFrom::TOptions ControlFromOpts;

        std::unordered_set<std::string> trustHeaders;

        void init(const ptree& pt);
    };

    struct AuthOpts {
        bool use;
        bool sslOnly;
        std::string sslMessage;
        bool useAfterTls;

        void init(const ptree& pt);
    };

    struct YarmOpts {
        bool useRpopAuth;
        std::string secret;

        void init(const ptree& pt);
    };

    struct RBLOpts {
        bool use = false;
        std::string clusterClientModuleName;
        std::vector<std::string> hosts;

        void init(const ptree& pt);
    };

    struct SPFOpts {
        bool use;
        unsigned int timeout;

        void init(const ptree& pt);
    };

    struct DMARCOpts {
        bool use;
        unsigned int timeout;
        void init(const ptree& pt);
    };

    struct LLOpts {
        using duration = yplatform::time_traits::duration;

        bool use;
        unsigned int attempts = 2;
        remote_point addr;
        duration timeout = yplatform::time_traits::seconds(2);
        duration connectTimeout = yplatform::time_traits::seconds(1);

        bool ignoreErrors;

        void init(const ptree& pt);
    };

    struct BigMlOpts {
        using duration = yplatform::time_traits::duration;

        bool use = false;
        bool add_recipients = false;
        std::string path;
        duration timeout = yplatform::time_traits::seconds(2);
        duration connectTimeout = yplatform::time_traits::seconds(1);

        void init(const ptree& pt);
    };

    struct AuthSettingsOpts {
        bool use = false;
        void init(const ptree& pt);
    };

    struct BlackBoxOpts {
        using duration = yplatform::time_traits::duration;

        std::set<std::string> oauthScopes;

        std::string pddEULApattern = "https://mail.yandex.%1%/for/%2%";
        std::set<std::string> knownCountries = {"ru", "by", "kz", "ua", "tr"};

        std::map<std::string, std::string> accessRestrictedLinksMap;
        const std::string defaultRestrictedComLink = "http://ya.cc/6lIh";

        bool allowUnknownRcpt = false;
        bool allowReceiveWithBadKarma = false;
        bool checkRcpt = false;
        bool checkSender = false;
        bool allowUnknownSender = false;
        std::string tempUserErrorReplyText;
        bool denyAuthForAssessors = false;

        void init(const ptree& pt);
    };

    struct RelayOpts {
        SmtpPoint addr;
        SmtpTimeouts timeouts;
    };

    struct Targeting {
        bool use = false;
        SmtpTimeouts timeouts = {1000ms, 1000ms, 1000ms, 0ms};
        std::optional<std::uint16_t> bypassPort;
        bool fallback = true;

        void init(const ptree& pt);
    };

    struct Mds {
        TCaseInsensitiveHashSet RemoveHeaders;

        void init(const ptree& pt);
    };

    struct DeliveryOpts {
        bool localTrustReject;
        std::optional<Targeting> localTargeting;
        std::optional<Targeting> fallbackTargeting;
        RelayOpts local;
        RelayOpts fallback;
        RelayOpts external;
        RoutingSettings routing;
        std::map<std::string, remote_point> mxCodeMap;
        std::unordered_set<std::string> removeHeaders;

        struct SenderDependentInfo {
            // List of headers to be removed when message is sent to foreign relay.
            // A sender-dependent override for global relay settings. Table is searched
            // by the envelope sender address and @domain.
            std::map<std::string, SmtpPoint> relaysMap;
            SmtpTimeouts timeouts;
        } senderDependent;

        void init(const ptree& pt);
    };

    struct DeliveryToSenderControl {
        bool use = false;
        bool checkSenderInBigMlSubscribers = false;

        void init(const ptree& pt);
    };

    struct DecyclerOpts {
        bool updateHeader = false;
        bool reject = false;
        std::size_t ttl = std::numeric_limits<std::size_t>::max();
        void init(const ptree& pt);
    };

    struct DsnOptions {
        int mode = 0;
        ymod_smtpclient::SmtpPoint relay;
        bool sign = false;
        dsn::Options composer;

        void init(const ptree& pt);
    };

    std::string clusterName;
    std::string hostName;
    std::string configFilename;
    std::string mydomain;
    std::string publicSuffixListFile;

    AuthOpts auth;
    YarmOpts yarm;

    SmtpConnectionOpts smtpOpts;
    MessageProcessingOpts msgOpts;

    BlackBoxOpts blackbox;
    LLOpts corpList;
    BigMlOpts bigMlOpts;
    AuthSettingsOpts authSettingsOpts;

    RBLOpts rbl;
    SPFOpts spf;
    DMARCOpts dmarc;

    avir_options avir;
    NSO::TOptions soOpts;

    Mds mds;
    DeliveryOpts delivery;

    DeliveryToSenderControl deliveryToSenderControl;

    AresOpts aresOpts;

    DecyclerOpts decyclerOpts;
    DsnOptions dsn;

    Options() = default;
    explicit Options(const ptree& pt);

    void reload(const ptree& pt);

private:
    void read(const ptree& pt);
};

extern std::shared_ptr<Options> gconfig;

extern const char *temp_error;

} // namespace NNwSmtp
