#include "interface.h"
#include "impl.h"

#include <yplatform/time_traits.h>
#include <yplatform/algorithm/leaky_bucket.h>
#include <yplatform/module.h>
#include <yplatform/reactor.h>

#include <boost/asio/io_context.hpp>

#include <atomic>
#include <memory>
#include <cstddef>

namespace NNwSmtp::NRateLimiter {

const auto LeakInterval = yplatform::time_traits::milliseconds(10);
const std::size_t SecondToLeakIntervalRatio = yplatform::time_traits::seconds(1) / LeakInterval;

TRequestsRateLimiterImpl::TRequestsRateLimiterImpl(TRequestsRateLimiterImplSettings settings)
    : Settings(std::move(settings))
    , RcptToLeakyBucket(Settings.RcptToLeakFactor,
        std::chrono::duration_cast<std::chrono::milliseconds>(Settings.RcptToLeakInterval).count())
    , DataLeakyBucket(Settings.DataLeakFactor, 
        std::chrono::duration_cast<std::chrono::milliseconds>(Settings.DataLeakInterval).count())
{
}

void TRequestsRateLimiterImpl::OnStartRcptTo() {
    std::unique_lock<std::mutex> lock(Mtx);
    RcptToLeakyBucket.add(SecondToLeakIntervalRatio);
}

void TRequestsRateLimiterImpl::OnEndRcptTo() {
}

bool TRequestsRateLimiterImpl::CheckLimitRcptTo() {
    std::unique_lock<std::mutex> lock(Mtx);
    return RcptToLeakyBucket.get() < Settings.RcptToLimit;
}

void TRequestsRateLimiterImpl::OnStartData() {
    std::unique_lock<std::mutex> lock(Mtx);
    DataLeakyBucket.add(SecondToLeakIntervalRatio);
}

void TRequestsRateLimiterImpl::OnEndData() {
}

bool TRequestsRateLimiterImpl::CheckLimitData() {
    std::unique_lock<std::mutex> lock(Mtx);
    return DataLeakyBucket.get() < Settings.DataLimit;
}

class TRequestsRateLimiterModule
    : public IRequestsRateLimiter
    , public yplatform::module {
public:
    void init(const yplatform::ptree& conf) {
        TRequestsRateLimiterImplSettings settings;

        settings.RcptToLimit = SecondToLeakIntervalRatio * conf.get<std::size_t>("rps");
        settings.RcptToLeakFactor = conf.get<std::size_t>("rps");
        settings.RcptToLeakInterval = LeakInterval;

        settings.DataLimit = SecondToLeakIntervalRatio * conf.get<std::size_t>("rps");
        settings.DataLeakFactor = conf.get<std::size_t>("rps");
        settings.DataLeakInterval = LeakInterval;

        Impl = std::make_shared<TRequestsRateLimiterImpl>(std::move(settings));
    }

    void OnStartRcptTo() override {
        Impl->OnStartRcptTo();
    }

    void OnEndRcptTo() override {
        Impl->OnEndRcptTo();
    }

    bool CheckLimitRcptTo() override {
        return Impl->CheckLimitRcptTo();
    }

    void OnStartData() override {
        Impl->OnStartData();
    }

    void OnEndData() override {
        Impl->OnEndData();
    }

    bool CheckLimitData() override {
        return Impl->CheckLimitData();
    }

private:
    std::shared_ptr<TRequestsRateLimiterImpl> Impl;
};

}

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(NNwSmtp::NRateLimiter::TRequestsRateLimiterModule)
