#include "checks.h"
#include "client.h"

#include <mail/nwsmtp/src/utils.h>

#include <yplatform/coroutine.h>

#include <memory>
#include <unordered_set>

namespace NNwSmtp::NRateSrv {

void AsyncCheckClient(
    TContextPtr ctx,
    const TIp& ip,
    TRateLimitCallback callback)
{
    auto client = std::make_shared<TClient<TUpdateRequestTag>>(std::move(ctx), true, true, std::move(callback));
    client->AddPartToRequest(ip.to_string(), "connections_from_ip", 1, ip);
    yplatform::spawn(std::move(client));
}

void AsyncCheckSender(
    TContextPtr ctx,
    std::string sender,
    std::string senderDefaultEmail,
    TRateLimitCallback callback)
{
    boost::to_lower(sender);
    auto senderDomain = GetDomainFromEmail(senderDefaultEmail);
    auto client = std::make_shared<TClient<TGetRequestTag>>(std::move(ctx), true, true, std::move(callback));
    client->AddPartToRequest(sender, "msgs_from_sndr", sender, senderDomain);
    client->AddPartToRequest(sender, "bytes_from_sndr", sender, senderDomain);
    yplatform::spawn(std::move(client));
}

void AsyncCheckRecipient(
    TContextPtr ctx,
    const TIp& ip,
    std::string sender,
    std::string senderDefaultEmail,
    std::string recipient,
    std::string recipientDefaultEmail,
    TRateLimitCallback callback)
{
    boost::to_lower(sender);
    boost::to_lower(recipient);
    auto senderDomain = GetDomainFromEmail(senderDefaultEmail);
    auto recipientDomain = GetDomainFromEmail(recipientDefaultEmail);
    auto client = std::make_shared<TClient<TGetRequestTag>>(std::move(ctx), true, true, std::move(callback));
    client->AddPartToRequest(recipient, "msgs_for_rcpt", recipient, recipientDomain);
    client->AddPartToRequest(recipient, "msgs_for_rcpt_from_ip", recipient, recipientDomain, ip);
    client->AddPartToRequest(recipient, "bytes_for_rcpt", recipient, recipientDomain);
    client->AddPartToRequest(recipient, "bytes_for_rcpt_from_ip", recipient, recipientDomain, ip);
    client->AddPartToRequest(recipient, "msgs_for_rcpt_from_sndr", sender, senderDomain, recipient, recipientDomain);
    yplatform::spawn(std::move(client));
}

void AsyncUpdateCounters(
    TContextPtr ctx,
    size_t bytes,
    const TIp& ip,
    std::string sender,
    std::string senderDefaultEmail,
    const TRcpts& recipients,
    TRateLimitCallback callback)
{
    std::unordered_map<std::string, std::string> goodRecipients;
    for (const auto& recipient : recipients) {
        if (recipient.m_delivery_status != check::CHK_ACCEPT || !recipient.need_update_limits) {
            continue;
        }

        auto recipientDomain = GetDomainFromEmail(recipient.m_default_email);
        if (!recipient.m_ml_uid.empty()) {
            goodRecipients.emplace(recipient.m_ml_uid, std::move(recipientDomain));
        } else if (!recipient.m_uid.empty()) {
            goodRecipients.emplace(recipient.m_uid, std::move(recipientDomain));
        } else {
            goodRecipients.emplace(boost::to_lower_copy(recipient.m_name), std::move(recipientDomain));
        }
    }

    if (goodRecipients.empty()) {
        return callback({});
    }

    boost::to_lower(sender);
    auto client = std::make_shared<TClient<TUpdateRequestTag>>(std::move(ctx), false, false, std::move(callback));
    auto senderDomain = GetDomainFromEmail(senderDefaultEmail);
    client->AddPartToRequest(sender, "msgs_from_sndr", 1, sender, senderDomain);
    client->AddPartToRequest(sender, "bytes_from_sndr", bytes, sender, senderDomain);

    for (const auto& [recipient, recipientDomain] : goodRecipients) {
        client->AddPartToRequest(recipient, "msgs_for_rcpt", 1, recipient, recipientDomain);
        client->AddPartToRequest(recipient, "msgs_for_rcpt_from_ip", 1, recipient, recipientDomain, ip);
        client->AddPartToRequest(recipient, "bytes_for_rcpt", bytes, recipient, recipientDomain);
        client->AddPartToRequest(recipient, "bytes_for_rcpt_from_ip", bytes, recipient, recipientDomain, ip);
        client->AddPartToRequest(
            recipient,
            "msgs_for_rcpt_from_sndr",
            1,
            sender,
            senderDomain,
            recipient,
            recipientDomain);
    }

    yplatform::spawn(std::move(client));
}

} // namespace NNwSmtp::NRateSrv
