#pragma once

#include "common.h"
#include "error_code.h"

#include <mail/nwsmtp/src/context.h>
#include <mail/nwsmtp/src/rcpts.h>

#include <string>

namespace NNwSmtp::NRateSrv {

using TRateLimitCallback = std::function<void(TErrorCode)>;

struct TRequest {
    std::string Sender;
    std::string SenderDefaultEmail;
};

struct IAsyncCheckSenderClient {
    virtual ~IAsyncCheckSenderClient() = default;

    using TCallback = TRateLimitCallback;
    virtual void Run(TContextPtr context, TRequest request, TCallback cb) = 0;
};

using TAsyncCheckSenderClientPtr = std::shared_ptr<IAsyncCheckSenderClient>;

struct TRcptRequest {
    TIp Ip;
    std::string Sender;
    std::string SenderDefaultEmail;
    std::string Recipient;
    std::string RecipientDefaultEmail;
};

struct IAsyncCheckRecipientClient {
    virtual ~IAsyncCheckRecipientClient() = default;

    using TCallback = TRateLimitCallback;
    virtual void Run(TContextPtr context, TRcptRequest request, TCallback cb) = 0;
};

using TAsyncCheckRecipientClientPtr = std::shared_ptr<IAsyncCheckRecipientClient>;

void AsyncCheckClient(
    TContextPtr ctx,
    const TIp& ip,
    TRateLimitCallback callback);

void AsyncCheckSender(
    TContextPtr ctx,
    std::string sender,
    std::string senderDefaultEmail,
    TRateLimitCallback callback);

void AsyncCheckRecipient(
    TContextPtr ctx,
    const TIp& ip,
    std::string sender,
    std::string senderDefaultEmail,
    std::string recipient,
    std::string recipientDefaultEmail,
    TRateLimitCallback callback);

void AsyncUpdateCounters(
    TContextPtr ctx,
    size_t bytes,
    const TIp& ip,
    std::string sender,
    std::string senderDefaultEmail,
    const TRcpts& recipients,
    TRateLimitCallback callback);

struct TAsyncCheckSenderClient : public IAsyncCheckSenderClient {
    void Run(TContextPtr context, TRequest request, TCallback cb) override {
        NRateSrv::AsyncCheckSender(context, request.Sender, request.SenderDefaultEmail, std::move(cb));
    }
};

struct TAsyncCheckRecipientClient : public IAsyncCheckRecipientClient {
    void Run(TContextPtr context, TRcptRequest request, TCallback cb) override {
        NRateSrv::AsyncCheckRecipient(context, request.Ip, request.Sender, request.SenderDefaultEmail,
            request.Recipient, request.RecipientDefaultEmail, std::move(cb));
    }
};

} // namespace NNwSmtp::NRateSrv
