#pragma once

#include "context.h"
#include "error.h"
#include "log.h"
#include "client.h"
#include "response.h"
#include "reflection.h"

#include <ymod_httpclient/cluster_client.h>
#include <mail/yreflection/include/yamail/data/deserialization/json_reader.h>

#include <boost/asio/io_context.hpp>
#include <boost/bind.hpp>

#include <string>

namespace NNwSmtp::NSO::NRBL {

using TClusterCall = ymod_httpclient::cluster_call;
using TClusterCallPtr = std::shared_ptr<TClusterCall>;

class THttpClient : public IClient {
public:
    THttpClient(TClusterCallPtr clusterCall, boost::asio::io_context& ioContext)
        : ClusterCall(clusterCall)
        , IoContext(ioContext)
    {}

    void AsyncCheck(TContextPtr ctx, std::string ip, TCallback cb) {
        std::string path = "/check?ip=" + ip + "&check=spamsource&service=nwsmtp";
        auto request = ymod_httpclient::request::GET(std::move(path));

        auto postOnExecutorCb = [ex = IoContext.get_executor(), cb = std::move(cb)]
            (TErrorCode ec, bool foundInSpam) mutable {
                boost::asio::post(ex,
                    std::bind(std::move(cb), std::move(ec), std::move(foundInSpam))
                );
            };

        namespace ph = std::placeholders;
        ClusterCall->async_run(ctx->CreateTaskContext("SO_RBL"), std::move(request),
            std::bind(&THttpClient::HandleResponse,
                ph::_1, ph::_2, std::move(postOnExecutorCb)
            )
        );
    }

    static void HandleResponse(TErrorCode ec, ymod_httpclient::response response, TCallback cb) {
        if (!ec && response.status == 200) {
            try {
                TSoResponse resp = yamail::data::deserialization::fromJson<TSoResponse>(response.body);
                bool foundInSpam = resp.Checks.SpamSource;
                cb(make_error_code(EError::Ok), foundInSpam);
            } catch (const std::exception&) {
                cb(make_error_code(EError::ResponseParseError), false);
            }
        } else { 
            TErrorCode errorCode;
            if (ec) {
                errorCode = std::move(ec);
            } else {
                errorCode = make_error_code(EError::Failed);
            }
            cb(std::move(errorCode), false);
        }
    }

private:
    TClusterCallPtr ClusterCall;
    boost::asio::io_context& IoContext;
};

using THttpClientPtr = std::shared_ptr<THttpClient>;

} // namespace NNwSmtp::NSO::NRBL
