#pragma once

#include <mail/nwsmtp/src/ratesrv/error_code.h>

#include <boost/system/error_code.hpp>

namespace NNwSmtp::NRcptTo {

enum class EError {
    Ok,
    MaillistIsReadonly,
    Reject,
    Discard,
    RejectByRateSrv,
    TooManyRecipients,
    TempFail
};

} // namespace NNwSmtp::NRcptTo

namespace boost::system {

template <>
struct is_error_code_enum<NNwSmtp::NRcptTo::EError> : std::true_type {};

} // namespace boost::system

namespace NNwSmtp::NRcptTo {

class TErrorCategory final : public boost::system::error_category {

public:
    const char* name() const noexcept override {
        return "nwsmtp_rcpt_to";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::Ok:
                return "Ok";
            case EError::MaillistIsReadonly:
                return "Maillist is read only";
            case EError::TempFail:
                return "Try again later";
            case EError::Reject:
                return "Reject";
            case EError::Discard:
                return "Discard";
            case EError::RejectByRateSrv:
                return "Reject by ratesrv";
            case EError::TooManyRecipients:
                return "Too many recipients";
        }
        return "Unknown error code: " + std::to_string(value);
    }

    bool equivalent(const boost::system::error_code& code, int condition) const noexcept override {
        switch (EError {condition}) {
            case EError::Discard:
                return code == NRateSrv::EError::Discard;
            case EError::RejectByRateSrv:
                return code == NRateSrv::EError::Reject;
            default:
                return boost::system::error_category::equivalent(code, condition);
        }
    }
};

inline const TErrorCategory& getErrorCategory() {
    static TErrorCategory errorCategory;
    return errorCategory;
}

inline auto make_error_code(EError ec) {
    return boost::system::error_code(static_cast<int>(ec), getErrorCategory());
}

inline auto make_error_condition(EError ec) {
    return boost::system::error_condition(static_cast<int>(ec), getErrorCategory());
}

} // namespace NNwSmtp::NRcptTo
