#include "rcpt_to_impl.h"
#include "utils.h"

#include <mail/nwsmtp/src/rcpt_to/expand_alias.h>
#include <mail/nwsmtp/src/big_ml/errors.h>
#include <mail/nwsmtp/src/ml/client_impl.h>
#include <mail/nwsmtp/src/aliases.h>
#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/src/bb_get_result.h>
#include <mail/nwsmtp/src/check.h>

#include <format>

namespace ph = std::placeholders;

namespace NNwSmtp::NRcptTo {

const std::string Where{"RCPT-TO-COMMAND"};

TRcptToCommandImpl::TRcptToCommandImpl(
    TContextPtr context,
    TRequest req,
    TCallback cb,
    TConfig config,
    NML::TClientPtr ml,
    NBigML::TClientPtr bigMl,
    NRateSrv::TAsyncCheckRecipientClientPtr rateSrvCheckRecipient,
    NBlackBox::TBBChecksPtr bbChecks,
    TRouterPtr router,
    boost::asio::io_context& ioContext)
    : Context(std::move(context))
    , Req(std::move(req))
    , Callback(std::move(cb))
    , Config(std::move(config))
    , MlClient(std::move(ml))
    , BigMlClient(std::move(bigMl))
    , RateSrvCheckRecipientClient(std::move(rateSrvCheckRecipient))
    , BBChecksClient(std::move(bbChecks))
    , RouterClient(std::move(router))
    , IoContext(ioContext)
{}

#include <boost/asio/yield.hpp>

void TRcptToCommandImpl::Run() {
    reenter(RunCoro) {
        NWLOG_L(notice, Where, "rcpt=" + Req.RcptTo);

        CheckRecipientAddressSyntax();
        if (ResponseErrorCode) {
            yield break;
        }

        PercentHack();
        if (ResponseErrorCode) {
            yield break;
        }

        if (RecipientsLimitExceeded(Config, Req.Envelope)) {
            ResponseErrorCode = EError::TooManyRecipients;
            Response.SmtpAnswer = "452 4.5.3 Error: too many recipients";
            yield break;
        }
        Rcpt.m_remote_ip = Req.RemoteIp.to_string();
        Rcpt.m_session_id = Context->GetSessionId();
        Rcpt.m_result = check::CHK_ACCEPT;
        Rcpt.RcptContext.Email = Req.RcptTo;
        Rcpt.RcptContext.IsAlias = gAliases.Contains(Req.RcptTo);

        SetRcptDsnNotifies(Req.Params);

        if (NeedCallRouter(Config, Rcpt)) {
            yield CallRouter();
            if (ResponseErrorCode) {
                yield break;
            }
        }

        if (NeedCheckBBForRcpt(Config, Rcpt)) {
            yield CheckBB();
            if (ResponseErrorCode) {
                yield break;
            }
        }

        yield CheckRateSrv();
        if (ResponseErrorCode) {
            yield break;
        }

        yield ExpandMaillist();
        if (ResponseErrorCode) {
            yield break;
        }
    }

    if (RunCoro.is_complete()) {
        if (ResponseErrorCode) {
            NWLOG_L_EC(notice, Where, "Error occurred", ResponseErrorCode);
        }

        boost::asio::post(IoContext,
            std::bind(std::move(Callback), std::move(ResponseErrorCode), std::move(Response))
        );
    }
}

void TRcptToCommandImpl::CheckRecipientAddressSyntax() {
    const auto& addr = Req.RcptTo;
    std::string::size_type dogPos = addr.find("@");
    if (dogPos == std::string::npos) {
        ResponseErrorCode = EError::Reject;
        Response.SmtpAnswer = "504 5.5.2 Recipient address rejected: need fully-qualified address";
        return;
    }

    if (dogPos == 0 || dogPos == addr.length() - 1) {
        ResponseErrorCode = EError::Reject;
        Response.SmtpAnswer = "501 5.1.3 Bad recipient address syntax";
        return;
    }

    if (Config.StrictAsciiRecipient && std::count_if(addr.begin(), addr.end(), is_invalid) > 0) {
        ResponseErrorCode = EError::Reject;
        Response.SmtpAnswer = "501 5.1.3 Bad recipient address syntax";
        return;
    }

    ResponseErrorCode = EError::Ok;
    Response.SmtpAnswer = std::string{};
}

void TRcptToCommandImpl::PercentHack() {
    std::string& addr = Req.RcptTo;
    std::string::size_type dogPos = addr.find("@");
    std::string::size_type percPos = addr.find("%");
    if (percPos != std::string::npos) {
        if (Config.PercentHack && percPos != 0) {
            addr = addr.substr(0, percPos) + "@" + addr.substr(percPos + 1, dogPos - percPos - 1);
        } else {
            ResponseErrorCode = EError::Reject;
            Response.SmtpAnswer = "501 5.1.3 Bad recipient address syntax";
            return;
        }
    }

    ResponseErrorCode = EError::Ok;
    Response.SmtpAnswer = std::string{};
}

void TRcptToCommandImpl::SetRcptDsnNotifies(std::map<std::string, std::string> params) {
    auto notify_param = params.find("notify");
    if (notify_param != params.end()) {
        NRcptTo::SetRcptDsnNotifies(Rcpt, notify_param->second);
    }
}

void TRcptToCommandImpl::CallRouter() {
    std::string local, domain;
    parse_email(Rcpt.RcptContext.Email, local, domain);
    RouterClient->asyncRoute(Context, domain,
        std::bind(&TRcptToCommandImpl::HandleCallRouter, shared_from_this(), ph::_1, ph::_2)
    );
}

void TRcptToCommandImpl::HandleCallRouter(TErrorCode ec, Router::Response resp) {
    if (ec) {
        ResponseErrorCode = NRcptTo::EError::Reject;
        Response.SmtpAnswer = "451 4.3.0 Try again later";
        return boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
    }

    Rcpt.RcptContext.DomainType = resp.domainType;
    boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
}

void TRcptToCommandImpl::CheckBB() {
    BBChecksClient->CheckRecipient(Context, removePlusFromEmail(Rcpt.RcptContext.Email), Rcpt.m_remote_ip, Config.CorpListOpts.use, true,
        std::bind(&TRcptToCommandImpl::HandleCheckBB, shared_from_this(), ph::_1, ph::_2)
    );
}

void TRcptToCommandImpl::HandleCheckBB(TErrorCode ec, NBlackBox::TResponse response)
{
    auto checkResult = NBlackBox::GetRcptResult(ec, response);

    Rcpt.m_result = checkResult.StatusCheck;
    Rcpt.m_answer = checkResult.SmtpResponse;
    Rcpt.RcptContext.BBData = std::move(checkResult.BlackBoxResult);

    if (NUtil::IsCheckSenderInRcpts(Config.ToSenderControl.use, Req.Envelope->m_sender_uid) &&
        (Req.Envelope->m_sender_uid == Rcpt.RcptContext.BBData.Uid)
    ) {
        Req.Envelope->is_sender_in_rcpts_ = true;
    }

    const auto& rcptContext = Rcpt.RcptContext;
    NWLOG_L(notice, "BB", "rcpt='" + rcptContext.Email + "', uid=" + rcptContext.BBData.Uid + ", suid=" +
        std::to_string(rcptContext.BBData.Suid) + ", status='" + checkResult.LogMsg + "', report='" + checkResult.SmtpResponse +
        "', remote_ip='" + Rcpt.m_remote_ip + "'");

    boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
}

void TRcptToCommandImpl::CheckRateSrv() {
    const auto& rcptContext = Rcpt.RcptContext;

    NRateSrv::TRcptRequest req {
        .Ip = Req.RemoteIp,
        .Sender = Req.Envelope->m_sender_uid.empty() ? Req.Envelope->m_sender : Req.Envelope->m_sender_uid,
        .SenderDefaultEmail = Req.Envelope->m_sender_default_email.empty() ? Req.Envelope->m_sender : Req.Envelope->m_sender_default_email,
        .Recipient = rcptContext.BBData.Uid.empty() ? rcptContext.Email : rcptContext.BBData.Uid,
        .RecipientDefaultEmail = rcptContext.BBData.DefaultEmail
    };

    RateSrvCheckRecipientClient->Run(Context, std::move(req), std::bind(&TRcptToCommandImpl::HandleRateSrv, shared_from_this(), ph::_1));
}

void TRcptToCommandImpl::HandleRateSrv(TErrorCode ec) {
    if (ec == make_error_condition(EError::RejectByRateSrv)) {
        NWLOG_L(notice, "RCPT", std::format("reject {} by RateSrv", Rcpt.RcptContext.Email));
        ResponseErrorCode = EError::RejectByRateSrv;
        Response.SmtpAnswer = "450 4.2.1 The recipient has exceeded message rate limit. Try again later.";
    } else if (ec == make_error_condition(EError::Discard)) {
        Rcpt.m_result = check_rcpt_t::CHK_DISCARD;
        NWLOG_L(notice, "RCPT", std::format("discard {} by RateSrv", Rcpt.RcptContext.Email));
        ResponseErrorCode = EError::Discard;
    }
    boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
}

void TRcptToCommandImpl::ExpandMaillist() {
    const auto& rcpt = Rcpt;
    const auto& rcptContext = rcpt.RcptContext;
    if (Config.CorpListOpts.use) {
        return ResolveCorpMaillist();
    } else if (Config.BigMlOpts.use && Config.BigMlOpts.add_recipients && rcptContext.BBData.IsMaillist) {
        return ResolveBigMlMaillist();
    } else if (rcpt.m_result == check::CHK_ACCEPT) {
        ExpandAlias(rcptContext, Req.Envelope, Config.ToSenderControl.use);
        ResponseErrorCode = EError::Ok;
        Response.SmtpAnswer.clear();
    } else {
        ResponseErrorCode = NRcptTo::EError::Reject;
        Response.SmtpAnswer = Rcpt.m_answer;
    }
    boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
}

void TRcptToCommandImpl::ResolveBigMlMaillist() {
    const std::string& from = Req.MailFrom;
    const std::string& to = removePlusFromEmail(Rcpt.RcptContext.Email);
    BigMlClient->Run(Context, {from, to},
        std::bind(&TRcptToCommandImpl::HandleBigMlMaillist, shared_from_this(), ph::_1, ph::_2));
}

void TRcptToCommandImpl::HandleBigMlMaillist(TErrorCode ec, NBigML::TResponse resp) {
    const auto& bbData = Rcpt.RcptContext.BBData;

    if (ec == NBigML::EC_PERMISSION_DENIED || ec == NBigML::EC_BAD_REQUEST) {
        NWLOG_L(notice, "RECV", "from=" + Req.MailFrom + ", read_only=" + std::to_string(
            ec == NBigML::EC_PERMISSION_DENIED) + ", result=" + ec.message());

        std::string error_msg = "Permanent error";
        if (ec == NBigML::EC_PERMISSION_DENIED) {
            error_msg = "Maillist is readonly";
        }
        ResponseErrorCode = EError::Reject;
        Response.SmtpAnswer = "530 5.7.2 " + error_msg;
        return boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
    }

    if (ec == NBigML::EC_NOT_FOUND || (!ec && resp.Subscribers.empty())) {
        if (Rcpt.m_result == check::CHK_ACCEPT) {
            // Recipient is not maillist and successfully resolved in BB,
            //  so add it to recipients list
            ExpandAlias(Rcpt.RcptContext, Req.Envelope, Config.ToSenderControl.use);
            ResponseErrorCode = EError::Ok;
            Response.SmtpAnswer.clear();
        } else {
            ResponseErrorCode = NRcptTo::EError::Reject;
            Response.SmtpAnswer = Rcpt.m_answer;
        }
        return boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
    }

    if (ec) {
        ResponseErrorCode = EError::TempFail;
        Response.SmtpAnswer = "451 4.3.0 Try again later";
        return boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
    }

    if (bbData.OrgId.empty()) {
        NWLOG_L(notice, "ORGID", std::format("status=empty_org_id, rcpt={}", Rcpt.RcptContext.Email));
    }

    for (const auto& [email, uid] : resp.Subscribers) {
        Req.Envelope->add_recipient(
            /*rcpt=           */ email,
            /*suid=           */ 0,
            /*is_alias=       */ true,
            /*uid=            */ uid,
            /*ml_uid=         */ bbData.Uid,
            /*default_email=  */ bbData.DefaultEmail,
            /*country_code=   */ "",
            /*domainType=     */ DomainType::LOCAL,
            /*notify_mode=    */ dsn::Options::NEVER,
            /*phone_confirmed=*/ false,
            /*is_maillist=    */ true);
    }

    Req.Envelope->m_no_local_relay = true;

    std::string result = "list expanded to " + std::to_string(resp.Subscribers.size()) + " recipients";
    NWLOG_L(notice, "RECV", "from=" + Req.MailFrom + ", read_only=" + std::to_string(false) + ", result=" + result);

    if (NUtil::IsCheckSenderInRcpts(Config.ToSenderControl.use, Req.Envelope->m_sender_uid) &&
        Config.ToSenderControl.checkSenderInBigMlSubscribers)
    {
        for (const auto& [_, uid] : resp.Subscribers) {
            if (Req.Envelope->m_sender_uid == uid) {
                Req.Envelope->is_sender_in_rcpts_ = true;
            }
        }
    }

    ResponseErrorCode = EError::Ok;
    Response.SmtpAnswer.clear();
    boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
}

void TRcptToCommandImpl::ResolveCorpMaillist() {
    const std::string& toEmail = Rcpt.RcptContext.Email;
    return MlClient->Run(Context, {toEmail}, std::bind(&TRcptToCommandImpl::HandleCorpMaillist, shared_from_this(), ph::_1, ph::_2));
}

void TRcptToCommandImpl::HandleCorpMaillist(TErrorCode ec, NML::TResponse resp) {
    const auto& rcptContext = Rcpt.RcptContext;
    const auto& bbData = rcptContext.BBData;

    if (!Config.CorpListOpts.ignoreErrors && ec && ec != list_client_ec::NOT_FOUND) {
        Response.SmtpAnswer = "451 4.3.0 Try again later";
        ResponseErrorCode = EError::TempFail;
        return boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
    }

    if ((ec && (Config.CorpListOpts.ignoreErrors || ec == list_client_ec::NOT_FOUND)) || resp.Subscribers.empty()) {
        if (Rcpt.m_result == check::CHK_ACCEPT) {
            // Recipient was successfully resolved in BB, so add it to recipients list
            Req.Envelope->add_recipient(
                /*rcpt=           */ rcptContext.Email,
                /*suid=           */ bbData.Suid,
                /*is_alias=       */ false,
                /*uid=            */ bbData.Uid,
                /*ml_uid=         */ bbData.Uid,
                /*default_email=  */ bbData.DefaultEmail,
                /*country_code=   */ bbData.Country,
                /*domainType=     */ rcptContext.DomainType,
                /*notify_mode=    */ rcptContext.NotifyMode,
                /*phone_confirmed=*/ bbData.PhoneConfirmed,
                /*is_maillist=    */ bbData.IsMaillist,
                /*regDate=        */ bbData.RegistrationDate);
            ResponseErrorCode = EError::Ok;
            Response.SmtpAnswer.clear();
        } else {
            ResponseErrorCode = NRcptTo::EError::Reject;
            Response.SmtpAnswer = Rcpt.m_answer;
        }
        return boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
    }

    std::string::size_type at = Req.MailFrom.find("@");
    std::string domain = (at != std::string::npos ? Req.MailFrom.substr(at + 1) : "");
    std::transform(domain.begin(), domain.end(), domain.begin(), ::tolower);

    bool inWriters = resp.WhoCanWrite.count(boost::to_lower_copy(Req.MailFrom));

    std::string result;
    bool fail = false;

    if (resp.Readonly && !inWriters) {
        ResponseErrorCode = EError::Reject;
        Response.SmtpAnswer = "530 5.7.2 This maillist is readonly for you!";

        result = "Maillist readonly and user not in writers";
        fail = true;
    } else if (resp.Internal && Req.Envelope->m_sender_uid.empty()) {
        ResponseErrorCode = EError::Reject;
        Response.SmtpAnswer = "530 5.7.2 This maillist is only for internal addresses!";

        result = "External user can't write to internal list";
        fail = true;
    } else {
        ResponseErrorCode = EError::Ok;
        Response.SmtpAnswer.clear();

        result = "list expanded to " + std::to_string(resp.Subscribers.size()) + " recipients";
    }

    NWLOG_L(notice, "RECV", "from=" + Req.MailFrom + ", domain=" + domain + ", read_only=" +
        std::to_string(resp.Readonly) + ", writers=" + std::to_string(inWriters) + ", internal=" +
        std::to_string(resp.Internal) + ", result=" + result);

    if (fail) {
        return boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
    }

    Req.Envelope->is_mailing_list_ |= bbData.IsCorpList;    // mark as mail list only if expanded
    if (Req.Envelope->is_mailing_list_ && Req.Envelope->ml_address_.empty()) {
        Req.Envelope->ml_address_ = rcptContext.Email;
    }

    for (const auto& subscriber : resp.Subscribers) {
        Req.Envelope->add_recipient(
            /*rcpt=             */ subscriber,
            /*suid=             */ 0,
            /*is_alias=         */ true,
            /*uid=              */ "",
            /*ml_uid=           */ bbData.Uid,
            /*default_email=    */ bbData.DefaultEmail,
            /*country_code=     */ "",
            /*domainType=       */ DomainType::UNKNOWN,
            /*notify_mode=      */ dsn::Options::NEVER);
    }
    Req.Envelope->m_no_local_relay |= !resp.Subscribers.empty();

    boost::asio::post(IoContext, std::bind(&TRcptToCommandImpl::Run, shared_from_this()));
}

#include <boost/asio/unyield.hpp>

} // namespace NNwSmtp::NRcptTo
