#pragma once

#include <mail/nwsmtp/src/context.h>
#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/utils.h>

#include <string>

namespace NNwSmtp::NDns {

static const std::string DNS{"DNS"};

template <class TResolver>
class TImpl {
public:
    explicit TImpl(TResolver& resolver)
        : Resolver(resolver)
    {}

    template <class TCallback>
    void AsyncResolveIp(
        TContextPtr context,
        const std::string& ip,
        TCallback&& callback)
    {
        auto ipAddress = boost::asio::ip::make_address(ip);

        Resolver.async_resolve_ptr(rev_order_str(ipAddress), TPtrCallback{
            [=, callback = std::forward<TCallback>(callback)](auto ec, auto iter) {
                if (ec) {
                    NWLOG_CTX(error, context, DNS, "error: " + ec.message());
                    return callback(std::move(ec), std::string{});
                }
                std::string host = (iter == TIteratorPtr() ? "" : *iter);
                // see about period at the end of fqdn:
                // https://tools.ietf.org/html/rfc1034#section-3.1
                if (!host.empty() && host.back() == '.') {
                    host.pop_back();
                }
                NWLOG_CTX(debug, context, DNS, "ip=" + ipAddress.to_string() + " resolved to host=" + host);
                return callback(std::move(ec), std::move(host));
            }});
    }

private:
    using TIteratorPtr = typename TResolver::iterator_ptr;
    using TPtrCallback = std::function<void(boost::system::error_code, TIteratorPtr)>;

    TResolver& Resolver;
};

}  // NNwSmtp::NDns
