#pragma once

#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/router/dns_router.h>

#include <yplatform/time_traits.h>

#include <boost/asio/yield.hpp>

namespace NNwSmtp {
namespace impl {

template <typename Resolver>
struct DnsRouterImpl: private boost::noncopyable {
    DnsRouterImpl(
        boost::asio::io_service& ios,
        const resolver_options& resolver_options,
        const DnsRouterSettings& settings
    )
        : strand(ios)
        , resolver(ios, resolver_options)
        , timer(ios)
        , settings(settings)
    {}

    boost::asio::io_service::strand strand;
    Resolver resolver;
    yplatform::time_traits::timer timer;
    const DnsRouterSettings& settings;
    unsigned short attempt = 0;
    Router::Handler handler;
    Router::Handler startHandler;
    std::string query;
    TContextPtr Context;
    yplatform::time_traits::time_point startedAt;
    yplatform::time_traits::duration reqDuration;
};

template <typename Resolver>
using ImplPtr = std::shared_ptr<DnsRouterImpl<Resolver>>;

template <typename ImplPtrType>
struct HandleStop {
    explicit HandleStop(const ImplPtrType& impl)
        : impl(impl)
    {}

    void operator()() {
        impl->reqDuration = yplatform::time_traits::clock::now() - impl->startedAt;
        try {
            impl->resolver.cancel();
            impl->timer.cancel();
        } catch (...) {
        }
    }

    ImplPtrType impl;
};

template <typename ImplPtrType>
struct HandleDone {
    explicit HandleDone(const ImplPtrType& impl)
        : impl(impl)
    {}

    void operator()(boost::system::error_code error, Router::Response response) {
        HandleStop<ImplPtrType> stop(impl);
        stop();
        auto handler = std::move(impl->startHandler);
        handler(std::move(error), std::move(response));
    }

    ImplPtrType impl;
};

template <typename ImplPtrType>
struct HandleResolve {
    explicit HandleResolve(const ImplPtrType& impl)
        : impl(impl)
    {
        impl->resolver.async_resolve_mx(impl->query, impl->strand.wrap(*this));
    }

    using Response = Router::Response;

    template <typename Iterator>
    void operator()(const boost::system::error_code& error, Iterator it) {
        if (error == boost::asio::error::operation_aborted) {
            return;
        }
        HandleDone<ImplPtrType> done(impl);

        if (error == boost::asio::error::host_not_found ||
            error == boost::asio::error::host_not_found_try_again)
        {
            done(error, Response(DomainType::INVALID));
        } else if (error) {
            done(error, Response(DomainType::UNKNOWN));
        } else if (it == Iterator()) {
            done(boost::system::error_code(), Response(DomainType::INVALID));
        } else {
            std::vector<typename std::iterator_traits<Iterator>::value_type> hosts(it, Iterator());
            std::sort(hosts.begin(), hosts.end());  // sort by priority
            auto priority = hosts.front().first;
            for (auto mx = hosts.begin(); mx != hosts.end() && mx->first == priority; ++mx) {
                if (impl->settings.myMXHosts.isMatch(mx->second)) {
                    return done(boost::system::error_code(),
                        Response(DomainType::LOCAL, mx->second));
                }
            }
            done(boost::system::error_code(), Response(DomainType::EXTERNAL, hosts.front().second));
        }
    }

    ImplPtrType impl;
};

template <typename ImplPtrType>
struct HandleTimeout {
    explicit HandleTimeout(const ImplPtrType& impl, yplatform::time_traits::duration duration)
        : impl(impl)
    {
        impl->timer.expires_from_now(duration);
        impl->timer.async_wait(impl->strand.wrap(*this));
    }

    void operator()(const boost::system::error_code& error) {
        if (error == boost::asio::error::operation_aborted) {
            return;
        }
        HandleDone<ImplPtrType> done(impl);
        done(make_error_code(boost::system::errc::timed_out), Router::Response());
    }

    ImplPtrType impl;
};

template <typename ImplPtrType>
struct HandleStart: private boost::asio::coroutine {
    explicit HandleStart(const ImplPtrType& impl)
        : impl(impl)
    {
        impl->strand.dispatch(*this);
    }

    void operator()(
        boost::system::error_code error = boost::system::error_code(),
        Router::Response response = Router::Response())
    {
        if (error == boost::asio::error::operation_aborted) {
            return;
        }
        reenter(this) {
            for (impl->attempt = 0; impl->attempt < impl->settings.attempts; ++impl->attempt) {
                yield {
                    impl->startedAt = yplatform::time_traits::clock::now();
                    impl->startHandler = *this;
                    HandleResolve<ImplPtrType> resolve(impl);
                    HandleTimeout<ImplPtrType> timeout(impl, impl->settings.timeout);
                    return;
                }
                if (error) {
                    NWLOG_CTX(error, impl->Context, "", "dns route domain=" + impl->query + ", attempt=" +
                        std::to_string(impl->attempt + 1) + ", error=" + error.message() + ", delay=" +
                        yplatform::time_traits::to_string(impl->reqDuration));
                } else {
                    NWLOG_CTX(notice, impl->Context, "", "dns route domain=" + impl->query + ", attempt=" +
                        std::to_string(impl->attempt + 1) + ", type=" + std::to_string(response.domainType) +
                        ", mx=" + response.mxHost + ", delay=" + yplatform::time_traits::to_string(
                        impl->reqDuration));
                }
                if (!error ||
                    error == boost::asio::error::host_not_found ||
                    error == boost::asio::error::host_not_found_try_again)
                {
                    error = boost::system::error_code();    // forbid retries
                    break;
                }
            }
            impl->startHandler = nullptr;
            impl->handler(std::move(error), std::move(response));
        }
    }

    ImplPtrType impl;
};

}   // namespace impl
}   // namespace NNwSmtp
