#include "router.h"
#include "impl/dns_router_impl.h"
#include "map_based_router.h"

#include <yadns/basic_dns_resolver.h>
#include <yadns/ares_resolver_service.h>

namespace NNwSmtp {

using Resolver = basic_dns_resolver<ares_resolver_service>;

TRouterPtr makeRouter(
    boost::asio::io_service& ios,
    RoutingSettings::Type type,
    const resolver_options& resolverOptions,
    const RoutingSettings& settings)
{
    if (type == RoutingSettings::Type::DNS) {
        return std::make_shared<DnsRouter<Resolver>>(ios, settings.dnsSettings, resolverOptions);
    } else if (type == RoutingSettings::Type::MAP) {
        return std::make_shared<MapBasedRouter>(ios, settings.mapSettings);
    } else {
        throw std::logic_error("invalid router type");
    }
}

RouterCall::RouterCall(
    boost::asio::io_service& ios,
    const resolver_options& resolverOptions,
    const RoutingSettings& settings
)
{
    primary = makeRouter(ios, settings.primary, resolverOptions, settings);
    if (settings.fallback != RoutingSettings::Type::NONE) {
        fallback = makeRouter(ios, settings.fallback, resolverOptions, settings);
    }
}

void RouterCall::asyncRoute(TContextPtr ctx, const std::string& domain, Handler handler) {
    primary->asyncRoute(ctx, domain, [impl = fallback, ctx, domain, h = std::move(handler)]
        (boost::system::error_code error, Router::Response response)
    {
        if (error && impl) {
            impl->asyncRoute(ctx, domain, std::move(h));
        } else {
            h(std::move(error), std::move(response));
        }
    });
}

}   // namespace NNwSmtp
