#pragma once

#include "errors.h"

#include <mail/nwsmtp/src/context.h>
#include <mail/nwsmtp/src/options.h>
#include <mail/nwsmtp/src/utils.h>

#include <butil/http/headers.h>
#include <yamail/data/deserialization/json_reader.h>
#include <ymod_httpclient/url_encode.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

BOOST_FUSION_DEFINE_STRUCT((NNwSmtp)(NSettings), TParametersSingleSettings,
    (std::optional<std::string>, enable_imap_auth_plain)
)

BOOST_FUSION_DEFINE_STRUCT((NNwSmtp)(NSettings), TProfileSingleSettings,
    (std::optional<std::string>, enable_pop)
    (std::optional<std::string>, enable_imap)
)

BOOST_FUSION_DEFINE_STRUCT((NNwSmtp)(NSettings), TParameters,
    (std::optional<NNwSmtp::NSettings::TParametersSingleSettings>, single_settings)
)

BOOST_FUSION_DEFINE_STRUCT((NNwSmtp)(NSettings), TProfile,
    (std::optional<NNwSmtp::NSettings::TProfileSingleSettings>, single_settings)
)

BOOST_FUSION_DEFINE_STRUCT((NNwSmtp)(NSettings), TSettings,
    (std::optional<NNwSmtp::NSettings::TParameters>, parameters)
    (std::optional<NNwSmtp::NSettings::TProfile>, profile)
)

BOOST_FUSION_DEFINE_STRUCT((NNwSmtp)(NSettings), TSettingsResponse,
    (std::optional<NNwSmtp::NSettings::TSettings>, settings)
)

namespace NNwSmtp::NSettings {

static const std::string SETTINGS{"SETTINGS"};

struct TAuthSettings {
    bool EnablePop{false};
    bool EnableImap{false};
    bool EnableImapAuthPlain{false};
};

static inline bool operator==(const TAuthSettings& left, const TAuthSettings& right) {
    return std::tie(left.EnablePop, left.EnableImap, left.EnableImapAuthPlain) ==
        std::tie(right.EnablePop, right.EnableImap, right.EnableImapAuthPlain);
}

using TCallback = std::function<void(boost::system::error_code, TAuthSettings)>;
using TUid = std::string;

class TSettingsClientImpl {
public:
    TSettingsClientImpl(
            TContextPtr context,
            TUid uid,
            TCallback callback,
            NUtil::TGetTvm2Module getTvm2Module,
            NUtil::TGetClusterClient getClusterClient)
        : Context{std::move(context)}
        , Uid{std::move(uid)}
        , Callback{std::move(callback)}
        , ClusterClient{getClusterClient()}
    {
        Tvm2Module = getTvm2Module();
    }

    using TYieldCtx = yplatform::yield_context<TSettingsClientImpl>;
    void operator()(TYieldCtx yieldCtx, boost::system::error_code errorCode = {},
            yhttp::response httpResponse = {})
    {
        try {
            reenter(yieldCtx) {
                yield ClusterClient->async_run(
                    Context->CreateTaskContext(SETTINGS + ":"),
                    MakeRequest(),
                    yieldCtx);

                if (errorCode) {
                    LogError(errorCode, MakeHttpStatusMessage(std::move(httpResponse)));
                } else if (httpResponse.status == 200) {
                    AuthSettings = ParseResponse(std::move(httpResponse.body));
                    NWLOG_L(notice, SETTINGS, "stat=ok");
                } else if ((httpResponse.status / 100) != 5) {
                    errorCode = make_error_code(EC_NON_RETRYABLE_STATUS);
                    LogError(errorCode, MakeHttpStatusMessage(std::move(httpResponse)));
                } else {
                    errorCode = make_error_code(EC_RETRIES_EXCEEDED);
                    LogError(errorCode, MakeHttpStatusMessage(std::move(httpResponse)));
                }
            }
        } catch (EErrorCode ec) {
            errorCode = make_error_code(ec);
            LogError(errorCode);
            return Callback(std::move(errorCode), {});
        } catch (const std::exception& ex) {
            errorCode = make_error_code(EC_EXCEPTION);
            LogException(errorCode, ex);
            return Callback(std::move(errorCode), {});
        }

        if (yieldCtx.is_complete()) {
            Callback(std::move(errorCode), std::move(AuthSettings));
        }
    }

private:
    void LogError(boost::system::error_code errorCode, const std::string& message = {}) const {
        using namespace yplatform::time_traits;
        NWLOG_L(error, SETTINGS, "stat=error (" + errorCode.message() + (message.empty() ? "" : (": " +
            message)) + ")");
    }

    void LogException(boost::system::error_code errorCode, const std::exception& exception) const {
        using namespace yplatform::time_traits;
        NWLOG_L_EXC(error, SETTINGS, "stat=error (" + errorCode.message() + ")", exception);
    }

    std::string GetTvmServiceTicket() const {
        const std::string serviceName{"settings"};
        std::string ticket;
        const auto errorCode{Tvm2Module->get_service_ticket(serviceName, ticket)};
        if (errorCode) {
            NWLOG_L(error, SETTINGS, "Failed to get TVM2 service ticket (" + errorCode.message() + ")");
            throw EC_TVM_SERVICE_TICKET_ERROR;
        }

        return ticket;
    }

    std::string MakeHeaders(std::string tvmServiceTicket) const {
        http::headers headers;
        headers.add("X-Ya-Service-Ticket", tvmServiceTicket);
        return headers.format();
    }

    yhttp::request MakeRequest() const {
        std::string url{"/get" +  yhttp::url_encode({{"uid", Uid},
            {"settings_list", "enable_pop\renable_imap\renable_imap_auth_plain"}, {"service", "smtp"}})};
        return yhttp::request::GET(std::move(url), MakeHeaders(GetTvmServiceTicket()));
    }

    std::string MakeHttpStatusMessage(yhttp::response httpResponse) {
        return "status=" + std::to_string(httpResponse.status) + ", reason=" +
            std::move(httpResponse.reason) + ", body=" + std::move(httpResponse.body);
    }

    TAuthSettings ParseResponse(std::string httpBody) {
        const auto settingsResponse{yamail::data::deserialization::fromJson<TSettingsResponse>(httpBody)};
        if (!settingsResponse.settings) {
            NWLOG_L(error, SETTINGS, "settings field not found");
            throw EC_BAD_RESPONSE;
        }

        const auto& parameters{settingsResponse.settings->parameters};
        if (!parameters) {
            NWLOG_L(error, SETTINGS, "parameters field not found");
            throw EC_BAD_RESPONSE;
        }

        const auto& profile{settingsResponse.settings->profile};
        if (!profile) {
            NWLOG_L(error, SETTINGS, "profile field not found");
            throw EC_BAD_RESPONSE;
        }

        const auto& parametersSingleSettings{parameters->single_settings};
        if (!parametersSingleSettings) {
            NWLOG_L(error, SETTINGS, "parameters single_settings field not found");
            throw EC_BAD_RESPONSE;
        }

        const auto& profileSingleSettings{profile->single_settings};
        if (!profileSingleSettings) {
            NWLOG_L(error, SETTINGS, "profile single_settings field not found");
            throw EC_BAD_RESPONSE;
        }

        const auto& enableImapAuthPlain{parametersSingleSettings->enable_imap_auth_plain};
        if (!enableImapAuthPlain) {
            NWLOG_L(error, SETTINGS, "enable_imap_auth_plain field not found");
        }

        const auto& enablePop{profileSingleSettings->enable_pop};
        if (!enablePop) {
            NWLOG_L(error, SETTINGS, "enable_pop field not found");
            throw EC_BAD_RESPONSE;
        }

        const auto& enableImap{profileSingleSettings->enable_imap};
        if (!enableImap) {
            NWLOG_L(error, SETTINGS, "enable_imap field not found");
            throw EC_BAD_RESPONSE;
        }

        return {!enablePop->empty(), !enableImap->empty(),
            !(enableImapAuthPlain.value_or(std::string{})).empty()};
    }

private:
    TContextPtr Context;
    TUid Uid;
    TCallback Callback;
    std::shared_ptr<ymod_tvm::tvm2_module> Tvm2Module;
    std::shared_ptr<ymod_httpclient::cluster_call> ClusterClient;
    TAuthSettings AuthSettings;
};

class TSettingsClient {
public:
    void GetAuthSettings(
        TContextPtr context,
        TUid uid,
        boost::asio::io_context& ioContext,
        TCallback callback,
        NUtil::TGetTvm2Module getTvm2Module = NUtil::MakeGetTvm2Module("tvm"),
        NUtil::TGetClusterClient getClusterClient = NUtil::MakeGetClusterClient("settings_client")) const
    {
        auto impl{std::make_shared<TSettingsClientImpl>(std::move(context), std::move(uid),
            std::move(callback), std::move(getTvm2Module), std::move(getClusterClient))};
        yplatform::spawn(ioContext.get_executor(), std::move(impl));
    }
};

} // namespace NNwSmtp::NSettings
