#pragma once

#include "settings_authorization.h"

#include <mail/nwsmtp/src/blackbox/bb_checks.h>
#include <mail/nwsmtp/src/bb_get_result.h>
#include <mail/nwsmtp/src/blackbox/error_code.h>

#include <mail/nwsmtp/src/settings/settings_client.h>

#include <util/string/cast.h>

namespace NNwSmtp {

namespace {

static const std::string SETTINGS_AUTHORIZATION{"SETTINGS_AUTHORIZATION"};

inline bool AuthAllowedBySettings(TContextPtr context, const NSettings::TAuthSettings& authSettings,
    NAuth::EAuthMethod authMethod, bool appPasswordEnabled)
{
    if (authSettings.EnablePop) {
        return true;
    }

    if (!authSettings.EnableImap) {
        NWLOG_CTX(notice, context, SETTINGS_AUTHORIZATION, "stat=rejected by pop and imap off");
        return false;
    }

    const auto authPlainRequested = (authMethod == NAuth::EAuthMethod::PLAIN) ||
        (authMethod == NAuth::EAuthMethod::LOGIN);
    if (!authSettings.EnableImapAuthPlain && authPlainRequested && !appPasswordEnabled) {
        NWLOG_CTX(notice, context, SETTINGS_AUTHORIZATION, "stat=rejected by plain auth off");
        return false;
    }

    return true;
}

}

template<typename TSettingsClient>
class TSettingsAuthorization : public std::enable_shared_from_this<TSettingsAuthorization<TSettingsClient>>
{
public:
    using TCallback = NSettingsAuthorization::TCallback;
    TSettingsAuthorization(
        boost::asio::io_context& ioContext,
        TContextPtr context,
        Options::AuthSettingsOpts authSettingsOpts,
        NBlackBox::TBBChecksPtr bbChecks,
        std::shared_ptr<TSettingsClient> settingsClient,
        TCallback callback)
        : IoContext{ioContext}
        , Context{std::move(context)}
        , AuthSettingsOpts(std::move(authSettingsOpts))
        , BBChecks{std::move(bbChecks)}
        , SettingsClient{std::move(settingsClient)}
        , Callback{std::move(callback)}
    {}

    void Start(TAuthData info) {
        auto handler = [self = this->shared_from_this(), this, info](auto ec, auto authResultData) mutable {
            boost::asio::post(IoContext, std::bind(&TSettingsAuthorization::HandleAuthResultData, self,
                std::move(ec), std::move(authResultData), std::move(info)));
        };

        BBChecks->CheckAuth(Context, std::move(info), std::move(handler));
    }

private:
    void HandleAuthSettings(boost::system::error_code errorCode, NSettings::TAuthSettings authSettings,
        TErrorCode ec, NBlackBox::TResponse response, TAuthData info) {
        auto authMethod = NAuth::EAuthMethod::INVALID;
        if (!TryFromString(info.Method, authMethod)) {
            authMethod = NAuth::EAuthMethod::INVALID;
        }

        if (errorCode) {
            NWLOG_L_EC(notice, SETTINGS_AUTHORIZATION, "stat=bypassed by Settings error", errorCode);
            Report(ec, std::move(response));
        } else if (AuthAllowedBySettings(Context, authSettings, authMethod, response.AppPasswordEnabled)) {
            NWLOG_L(notice, SETTINGS_AUTHORIZATION, "stat=accepted");
            Report(ec, std::move(response));
        } else {
            Report(NBlackBox::EError::NoAccessRights, std::move(response));
        }
    }

    void HandleAuthResultData(TErrorCode ec, NBlackBox::TResponse response, TAuthData info) {
        auto checkResult = NBlackBox::GetAuthResult(ec, response);

        if (!ec) {
            if (AuthSettingsOpts.use) {
                auto settingsClientCallback = [self = this->shared_from_this(),
                    info = std::move(info), ec = std::move(ec),
                    response = response, this](auto errorCode, auto authSettings)
                {
                    HandleAuthSettings(std::move(errorCode), std::move(authSettings),
                        std::move(ec), std::move(response), std::move(info));
                };

                SettingsClient->GetAuthSettings(Context, response.Uid, IoContext, std::move(settingsClientCallback));
            } else {
                NWLOG_L(notice, SETTINGS_AUTHORIZATION, "stat=bypassed by options");
                Report(ec, std::move(response));
            }
        } else {
            NWLOG_L_EC(notice, SETTINGS_AUTHORIZATION, "stat=bypassed by BB status", ec);
            Report(ec, std::move(response));
        }
    }

    void Report(TErrorCode ec, NBlackBox::TResponse response) {
        auto handler = [ec = std::move(ec), response = std::move(response), self = this->shared_from_this(), this] {
            Callback(ec, response);
        };

        boost::asio::post(IoContext, std::move(handler));
    }

    boost::asio::io_context& IoContext;
    TContextPtr Context;
    Options::AuthSettingsOpts AuthSettingsOpts;


    NBlackBox::TBBChecksPtr BBChecks;

    TAuthData Info;
    std::shared_ptr<TSettingsClient> SettingsClient;
    TCallback Callback;
};

using TSettingsAuthorizationPtr = std::shared_ptr<TSettingsAuthorization<NSettings::TSettingsClient>>;

namespace NSettingsAuthorization {

class TSettingsAuthorizationImpl : public ISettingsAuthorization {
public:
    TSettingsAuthorizationImpl(boost::asio::io_context& ioContext,
        Options::AuthSettingsOpts authSettingsOpts,
        std::shared_ptr<NSettings::TSettingsClient> settingsClient,
        NBlackBox::TBBChecksPtr bbChecks)
        : IoContext(ioContext)
        , AuthSettingsOpts(std::move(authSettingsOpts))
        , SettingsClient(settingsClient)
        , BBChecks(bbChecks)
    {}

    void Run(TContextPtr context, TRequest request, TCallback cb) override {
        auto client = std::make_shared<TSettingsAuthorization<NSettings::TSettingsClient>>(
            IoContext,
            std::move(context),
            AuthSettingsOpts,
            BBChecks,
            SettingsClient,
            std::move(cb));
        client->Start(std::move(request.Data));
    }

private:
    boost::asio::io_context& IoContext;
    Options::AuthSettingsOpts AuthSettingsOpts;
    std::shared_ptr<NSettings::TSettingsClient> SettingsClient;

    NBlackBox::TBBChecksPtr BBChecks;

};

} // namespace NSettingsAuthorization

} // namespace NNwSmtp
