#include "error_code.h"
#include "smtp_client.h"
#include "log.h"
#include "utils.h"
#include "smtp_targeting.h"

#include <yplatform/time_traits.h>
#include <util/random/random.h>

#include <boost/algorithm/string/trim.hpp>
#include <boost/format.hpp>

namespace NNwSmtp::SmtpClient {

namespace p = std::placeholders;

static const std::string SMTP_TARGETING{"SMTP-TARGETING"};

void TSmtpClient::HandleSend(
    TCallback callback,
    TRequestContext requestContext,
    TErrorCode errc,
    ymod_smtpclient::Response response
) {
    if (errc == ymod_smtpclient::error::TaskCancelled) {
        return callback(EError::TempFail);
    }

    TErrorCode result = EError::Ok;

    auto& rcpts = requestContext.Rcpts;
    const auto& protoName = requestContext.ProtoName;
    auto& envelope = requestContext.Envelope;
    const auto& host = requestContext.Host;
    const auto& realHost = requestContext.RealHost;
    const auto& startedAt = requestContext.StartedAt;
    const auto& sessionId = requestContext.SessionId;

    for (auto rcpt = rcpts.begin(); rcpt != rcpts.end(); ++rcpt) {
        check::chk_status status = check::CHK_TEMPFAIL;
        std::ostringstream remote;

        if (!errc || errc == ymod_smtpclient::error::PartialSend) {
            auto it = response.rcpts.find(rcpt->m_name);
            const auto& resp = (it == response.rcpts.end()) ? *response.session : it->second;
            status = MakeCheckStatus(resp.replyCode);
            remote << resp;
        } else if (response.session) {
            status = MakeCheckStatus(response.session->replyCode);
            if (status == check::CHK_ACCEPT) {  // impossible if error occured
                status = check::CHK_TEMPFAIL;
            }
            remote << *response.session;
        } else {
            result = EError::TempFail;
        }

        rcpt->m_delivery_status = status;
        rcpt->m_remote_answer = remote.str();

        if ((!result || result == EError::TempFail) &&
            rcpt->m_delivery_status != check::CHK_ACCEPT &&
            rcpt->m_delivery_status != check::CHK_DISCARD
        ) {
            result = rcpt->m_delivery_status == check::CHK_REJECT
                ? EError::Rejected
                : EError::TempFail;
        }

        using namespace yplatform::time_traits;
        const auto context{boost::make_shared<TContext>(
            sessionId,
            envelope->m_id,
            gconfig->clusterName,
            gconfig->hostName)};
        NWLOG_CTX(notice, context, "SEND-" + protoName, "to=" + rcpt->m_name + ", relay=" + host.host + ":" +
            std::to_string(host.port) + ", real=" + realHost.host + ":" + std::to_string(realHost.port) + ", delay=" +
            to_string(clock::now() - startedAt) + ", status=" + ((rcpt->m_delivery_status == check::CHK_ACCEPT) ?
            "sent" : "fault") + " (" + boost::trim_copy(rcpt->m_remote_answer) + ")");
    }

    envelope->copy_dlv_status(rcpts);
    callback(result);
}

void TSmtpClient::Start(
    TCallback callback,
    envelope_ptr envelope,
    const std::string& sessionId,
    const SmtpPoint& host,
    const std::string& protoName,
    const SmtpTimeouts& timeouts,
    const TDomainsSet& filter
) {
    envelope->init_all_recipients();
    envelope->cleanup_answers();

    DoStart(callback, envelope, sessionId, envelope->m_rcpts.begin(), envelope->m_rcpts.end(), envelope->m_sender,
        host, protoName, filter, timeouts);

}

void TSmtpClient::Start(
    TCallback callback,
    envelope_ptr envelope,
    envelope::rcpt_list_t& recipients,
    const std::string& sessionId,
    const std::string& sender,
    const SmtpPoint& host,
    const std::string& protoName,
    const SmtpTimeouts& timeouts,
    const std::optional<Options::Targeting>& targeting,
    const std::string& login,
    const std::string& password,
    bool oauth
) {
    for (auto& r : recipients) {
        r.m_delivery_status = check::CHK_TEMPFAIL;
        r.m_remote_answer.clear();
    }

    DoStart(callback, envelope, sessionId, recipients.begin(), recipients.end(), sender, host, protoName,
        TDomainsSet{}, timeouts, targeting, login, password, oauth);

}

void TSmtpClient::DoStart(
        TCallback callback,
        const envelope_ptr& envelope,
        const std::string& sessionId,
        const envelope::rcpt_list_t::iterator& rcptBeg,
        const envelope::rcpt_list_t::iterator& rcptEnd,
        const std::string& mailfrom,
        const SmtpPoint& host,
        const std::string& protoName,
        const TDomainsSet& filter,
        const SmtpTimeouts& timeouts,
        const std::optional<Options::Targeting>& targeting,
        const std::string& login,
        const std::string& password,
        bool oauth
) {

    envelope::rcpt_list_t rcpts;

    for (auto it = rcptBeg; it != rcptEnd; ++it) {
        std::string::size_type at = it->m_name.find('@');

        std::string domain = (at != std::string::npos ? it->m_name.substr(at + 1) : "");
        std::transform(domain.begin(), domain.end(), domain.begin(), ::tolower);

        if (filter.Empty() || filter.Contains(domain)) {
            rcpts.push_back(*it);
        }
    }

    if (rcpts.empty()) {
        return boost::asio::post(Io, std::bind(callback, TErrorCode{}));
    }

    auto request = SmtpClient::MakeRequest(host, login, password, oauth, mailfrom, envelope, rcpts);

    TRequestContext requestContext;

    requestContext.StartedAt = yplatform::time_traits::clock::now();
    requestContext.Envelope = envelope;
    requestContext.Rcpts = std::move(rcpts);
    requestContext.Mailfrom = mailfrom;
    requestContext.Host = host;
    requestContext.RealHost = host;
    requestContext.ProtoName = protoName;
    requestContext.SessionId = sessionId;

    auto ctx = boost::make_shared<TContext>(sessionId, envelope->m_id, gconfig->clusterName, gconfig->hostName);

    TCallback handler = [
        ex = boost::asio::get_associated_executor(callback, Io),
        callback = std::move(callback)
    ]
    (auto ec) {
        boost::asio::post(ex, std::bind(callback, ec));
    };

    if (targeting && targeting->use) {
        NWLOG_CTX(notice, ctx, SMTP_TARGETING, ("Targeting to " + host.host + ':' + std::to_string(host.port)));
        NSmtpTargeting::AsyncTargeting(
            Client->createSmtpSession(ctx, targeting->timeouts),
            host,
            [client = Client, ctx, request = std::move(request), requestContext = std::move(requestContext),
                timeouts, handler = std::move(handler), targeting = *targeting](auto ec, auto host) mutable {
                if (ec) {
                    NWLOG_CTX(error, ctx, SMTP_TARGETING,
                        ("Targeting failed with error '" + ymod_smtpclient::error::message(ec) + "'"));
                    if (!targeting.fallback) {
                        TSmtpClient::HandleSend(std::move(handler), std::move(requestContext), ec,
                            ymod_smtpclient::Response());

                        return;
                    }
                } else {
                    NWLOG_CTX(notice, ctx, SMTP_TARGETING, ("Successful targeting with result " + host));
                    request.address.host = host;
                    requestContext.RealHost.host = host;
                    if (targeting.bypassPort) {
                        request.address.port = *targeting.bypassPort;
                        requestContext.RealHost.port = *targeting.bypassPort;
                    }
                }

                client->asyncRun(ctx, std::move(request), SmtpClient::MakeOptions(timeouts),
                    std::bind(&TSmtpClient::HandleSend, std::move(handler), std::move(requestContext), p::_1, p::_2));
            }
        );
    } else {
        Client->asyncRun(ctx, std::move(request), SmtpClient::MakeOptions(timeouts),
            std::bind(&TSmtpClient::HandleSend, std::move(handler), std::move(requestContext), p::_1, p::_2));
    }
}

} // namespace NNwSmtp::SmtpClient
