#include "error_code.h"
#include "utils.h"

#include <mail/library/dsn/options.hpp>
#include <mail/nwsmtp/src/log.h>

#include <vector>

namespace NNwSmtp::SmtpClient {

namespace {

inline std::vector<ymod_smtpclient::NotifyMode> ExtractNotifyModes(std::int32_t mode) {
    std::vector<ymod_smtpclient::NotifyMode> result;
    if (mode == dsn::Options::NONE) {
        return result;  // return empty result
    }
    if (mode & dsn::Options::NEVER) {
        result.push_back(ymod_smtpclient::NotifyMode::Never);
        return result;  // return only one notify mode
    }
    if (mode & dsn::Options::SUCCESS) {
        result.push_back(ymod_smtpclient::NotifyMode::Success);
    }
    if (mode & dsn::Options::FAILURE) {
        result.push_back(ymod_smtpclient::NotifyMode::Failure);
    }
    if (mode & dsn::Options::DELAY) {
        result.push_back(ymod_smtpclient::NotifyMode::Delay);
    }
    return result;
}

} // namespace anonymous

ymod_smtpclient::Options MakeOptions(const ymod_smtpclient::Timeouts& timeouts) {
    ymod_smtpclient::Options opts;
    opts.timeouts = timeouts;
    return opts;
}

ymod_smtpclient::Request MakeRequest(
    const ymod_smtpclient::SmtpPoint& host,
    const std::string& login,
    const std::string& password,
    bool oauth,
    const std::string& mailfrom,
    const envelope_ptr& envelope,
    const envelope::rcpt_list_t& rcpts
) {
    ymod_smtpclient::RequestBuilder builder;
    builder.address(host);
    if (!login.empty()) {
        if (oauth) {
            builder.auth(ymod_smtpclient::AuthData::XOAUTH2(login, password));
        } else {
            builder.auth(ymod_smtpclient::AuthData::BEST(login, password));
        }
    }
    ymod_smtpclient::MailFrom from(mailfrom);
    if (!envelope->m_original_id.empty()) {
        from.envid = envelope->m_original_id;
    }
    builder.mailfrom(std::move(from));
    for (auto it = rcpts.begin(); it != rcpts.end(); ++it) {
        ymod_smtpclient::RcptTo rcpt(it->m_name);
        rcpt.notifyModes = ExtractNotifyModes(it->m_notify_mode);
        builder.addRcpt(std::move(rcpt));
    }
    builder.message(envelope->altered_message_);
    return builder.release();
}

check::chk_status MakeCheckStatus(std::uint32_t code) {
    if ((code >= 200) && (code < 300)) {
        return check::CHK_ACCEPT;
    } else if ((code >= 400) && (code < 500)) {
        return check::CHK_TEMPFAIL;
    } else if ((code >= 500) && (code < 600)) {
        return check::CHK_REJECT;
    }

    return check::CHK_TEMPFAIL;
}

} // namespace NNwSmtp::SmtpClient
