#include "session_impl.h"
#include "auth.h"
#include "includes.h"

#include <ymod_smtpserver/response.h>

#include <boost/algorithm/string.hpp>

namespace NNwSmtp::NSmtpServer {

namespace {

ymod_smtpserver::Response ConvertStringToYmodSmtpResponse(const std::string& answer) {
    std::uint32_t smtpCode = 0;
    std::uint32_t enhancedCode = 0;
    if (
        answer.size() >= 10 &&
        boost::conversion::try_lexical_convert(answer.substr(0, 3), smtpCode) &&
        boost::conversion::try_lexical_convert(boost::erase_all_copy(answer.substr(4, 5), "."), enhancedCode)
    ) {
        std::string response;
        if (answer.size() > 10) {
            response = answer.substr(10);
        }
        return ymod_smtpserver::Response(smtpCode, std::move(response), ymod_smtpserver::EnhancedStatusCode(enhancedCode));
    } else {
        return ymod_smtpserver::Response(451, "Try again later", ymod_smtpserver::EnhancedStatusCode(430));
    }
}

}

namespace ph = std::placeholders;

TSessionImpl::TSessionImpl(ConnectionPtr connection, boost::asio::io_context& ioContext)
    : ymod_smtpserver::Session(connection)
    , Connection(connection)
    , IoContext(ioContext)
    , SessionTimer(ioContext)
{
    auto httpReactor = yplatform::global_reactor_set->get("http");
    auto nwsmtpReactor = yplatform::global_reactor_set->get("global");
    auto smtpReactor = yplatform::global_reactor_set->get("smtp");

    auto blackboxClient = std::make_shared<NBlackBox::NClient::TClientImpl>(
        yplatform::find<yhttp::cluster_client, std::shared_ptr>("blackbox_client"),
        yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("tvm"),
        *nwsmtpReactor->io()
    );
    auto bbChecks = std::make_shared<NBlackBox::TBBChecksImpl>(
        NBlackBox::MakeConfig(gconfig),
        std::move(blackboxClient)
    );
    auto spfCheckClient = std::make_shared<NSPF::TSPFCheckClient<spf_check>>(
        NSPF::MakeConfig(gconfig),
        *httpReactor->io()
    );
    auto settingsAuthorization = std::make_shared<NSettingsAuthorization::TSettingsAuthorizationImpl>(
        *httpReactor->io(),
        gconfig->authSettingsOpts,
        std::make_shared<NSettings::TSettingsClient>(),
        bbChecks
    );
    auto asyncCheckSenderClient = std::make_shared<NRateSrv::TAsyncCheckSenderClient>();
    auto mailFromConfig = NMailFromCommand::MakeConfig(gconfig);
    auto mailFromCmd = std::make_shared<NMailFromCommand::TMailFromCommand>(
        std::move(mailFromConfig),
        spfCheckClient,
        bbChecks,
        settingsAuthorization,
        asyncCheckSenderClient,
        *nwsmtpReactor->io());

    auto rateSrvCheckRecipient = std::make_shared<NRateSrv::TAsyncCheckRecipientClient>();

    std::shared_ptr<RouterCall> router;
    if (NUtil::IsRouting()) {
        router = std::make_shared<RouterCall>(
            *httpReactor->io(),
            gconfig->aresOpts.resolverOptions,
            gconfig->delivery.routing);
    };

    auto bigMLClient = std::make_shared<NBigML::TClient>(
        gconfig->bigMlOpts,
        yplatform::find<ymod_httpclient::cluster_call, std::shared_ptr>("big_ml_client"),
        *nwsmtpReactor->io()
    );

    auto mlClient = std::make_shared<NML::TClient>(
        gconfig->corpList,
        yplatform::find<yhttp::simple_call, std::shared_ptr>("http_client"),
        *nwsmtpReactor->io()
    );

    auto rcptToCmd = std::make_shared<NRcptTo::TRcptToCommand>(
        NRcptTo::MakeConfig(gconfig),
        mlClient,
        bigMLClient,
        rateSrvCheckRecipient,
        bbChecks,
        router,
        *nwsmtpReactor->io()
    );

    auto smtpClient = std::make_shared<SmtpClient::TSmtpClient>(
        yplatform::find<ymod_smtpclient::Call, std::shared_ptr>("smtp_client"),
        *nwsmtpReactor->io()
    );

    auto soCheckClient = std::make_shared<NSO::TSOClient>(
        yplatform::find<yhttp::cluster_client, std::shared_ptr>(gconfig->soOpts.SoClient),
        gconfig->soOpts.SoClient == "so_out_client" ? NSO::ESOType::SOOut : NSO::ESOType::SOIn,
        *nwsmtpReactor->io()
    );

    auto rblHttpClient = yplatform::find<ymod_httpclient::cluster_client, std::shared_ptr>("rbl_http_client");
    SoRblChecker = std::make_shared<NNwSmtp::NSO::NRBL::THttpClient>(rblHttpClient, *nwsmtpReactor->io());

    auto avirCheckClient = std::make_shared<NAvir::TAvirCheckClient>(
        std::make_shared<avir_client>(*nwsmtpReactor->io(), gconfig->avir.client_opts),
        *nwsmtpReactor->io()
    );

    auto recognizer = yplatform::find<TRecognizer, std::shared_ptr>("recognizer");

    auto yarmClient = std::make_shared<NYarm::TClientImpl>(
        gconfig->yarm,
        yplatform::find<ymod_httpclient::cluster_call, std::shared_ptr>("yarm_client"),
        *nwsmtpReactor->io()
    );

    auto delivery = std::make_shared<NAsyncDlv::TAsyncDelivery>(
        smtpClient,
        soCheckClient,
        avirCheckClient,
        recognizer,
        yarmClient,
        router,
        *nwsmtpReactor->io()
    );

    auto resolverModule = yplatform::find<NDns::TResolver, std::shared_ptr>("resolver");
    ResolverClient = std::make_shared<NDns::TResolverClient>(std::move(resolverModule), *nwsmtpReactor->io());

    Nwsmtp = std::make_shared<NNwSmtp::TNwsmtpImpl>(
        std::move(mailFromCmd),
        std::move(rcptToCmd),
        std::move(settingsAuthorization),
        std::move(delivery),
        *nwsmtpReactor->io());

    Auth = std::make_shared<NAuth::TAuthCredentialsExtractorImpl>();

    RateLimiter = yplatform::find<NRateLimiter::IRequestsRateLimiter, std::shared_ptr>("smtp_rate_limiter");
}

void TSessionImpl::start() {
    NWLOG_CTX(notice, Nwsmtp->GetContext(), "SERVER_INFO", "This session is handled by new nwmstp (based on ymod_smtpserver)");
    RemoteAddr = NUtil::ToIpV4IfPossible(connection->remoteAddr());
    Nwsmtp->SetRemoteIp(RemoteAddr);

    if (IsSessionTimeLimitEnabled()) {
        SessionTimer.expires_after(gconfig->smtpOpts.sessionTimeLimit.timeout);
        SessionTimer.async_wait([self = shared_from_this(), this](TErrorCode ec) {
            if (!ec) {
                NWLOG_CTX(notice, Nwsmtp->GetContext(), "RECV", fmt::format("connection will be closed because of session timer", RemoteHostName, RemoteAddr.to_string()));
                SessionTimeExpired.store(true);
            }
        });
    }

    ResolverClient->ResolveIp(Nwsmtp->GetContext(), RemoteAddr.to_string(),
        [self = shared_from_this(), &ioContext = IoContext](TErrorCode ec, std::string hostName) {
            boost::asio::post(ioContext,
                std::bind(&TSessionImpl::HandleIpResolve, self, std::move(ec), std::move(hostName))
            );
        }
    );
}

void TSessionImpl::HandleIpResolve(TErrorCode ec, std::string hostName) {
    if (!ec) {
        RemoteHostName = hostName;
        Nwsmtp->SetRemoteHostName(std::move(hostName));
    } else {
        RemoteHostName = "unknown";
        Nwsmtp->SetRemoteHostName("unknown");
    }
    NWLOG_CTX(notice, Nwsmtp->GetContext(), "RECV", fmt::format("connect from {} [{}] on [{}]:{}", hostName, RemoteAddr.to_string(), Connection->localAddr().to_string(), std::to_string(connection->localPort())));

    if (gconfig->rbl.use) {
        SoRblChecker->AsyncCheck(Nwsmtp->GetContext(), RemoteAddr.to_string(),
        [self = shared_from_this(), &ioContext = IoContext](TErrorCode ec, bool foundInSpam) {
            boost::asio::post(ioContext,
                std::bind(&TSessionImpl::HandleSoRbl, self, std::move(ec), std::move(foundInSpam))
            );
        });
    } else {
        CheckRateLimit();
    }
}

void TSessionImpl::HandleSoRbl(TErrorCode ec, bool foundInSpam) {
    if (ec) {
        NWLOG_EC_CTX(notice, Nwsmtp->GetContext(), "RBL", "Error occured", ec);
    }

    if (foundInSpam) {
        auto remoteIp = RemoteAddr.to_string();
        auto explain = fmt::format("Service unavailable; Client host [{}] blocked by spam statistics - see http://feedback.yandex.ru/?from=mail-rejects&subject={}", remoteIp, remoteIp);
        auto response = ymod_smtpserver::Response(554, std::move(explain), ymod_smtpserver::EnhancedStatusCode(571));
        NWLOG_CTX(notice, Nwsmtp->GetContext(), "RECV", fmt::format("reject: CONNECT FROM {} blocked by spam statistics", remoteIp));
        return WriteResponseWithUniqId(std::move(response), [this, self = shared_from_this()]() {
            return CloseConnection();
        });
    }

    CheckRateLimit();
}

void TSessionImpl::CheckRateLimit() {
    auto cb = [self = shared_from_this(), &ioContext = IoContext](TErrorCode ec) {
        boost::asio::post(ioContext, std::bind(&TSessionImpl::HandleCheckRateLimit, self, std::move(ec)));
    };
    return NNwSmtp::NRateSrv::AsyncCheckClient(Nwsmtp->GetContext(), RemoteAddr, std::move(cb));
}

void TSessionImpl::HandleCheckRateLimit(TErrorCode ec) {
    if (ec) {
        NWLOG_CTX(notice, Nwsmtp->GetContext(), "RECV", fmt::format("reject by RateSrv: CONNECT FROM {}", RemoteAddr.to_string()));
        auto response = ymod_smtpserver::Response(421, fmt::format("Error: too many connections from {}", RemoteAddr.to_string()), ymod_smtpserver::EnhancedStatusCode(470));

        return WriteResponseWithUniqId(std::move(response), [this, self = shared_from_this()]() {
            return CloseConnection();
        });
    }

    std::string addToResponse = !gconfig->smtpOpts.banner.empty() ? gconfig->smtpOpts.banner : "Ok";
    return WriteReadWithUniqId(ymod_smtpserver::Response(220, fmt::format("{} {}", boost::asio::ip::host_name(), std::move(addToResponse))));
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::Ehlo& ehlo) {
    if (!IsTransitionAllowed(CurrentState, SmtpState::Ehlo)) {
        return WriteReadWithUniqId(ymod_smtpserver::Response(503, "Bad sequence of commands.", ymod_smtpserver::EnhancedStatusCode(554)));
    }

    Nwsmtp->SetEhloHost(ehlo.name);
    std::ostringstream respText;
    respText << boost::asio::ip::host_name() << "\n" << "8BITMIME\n" << "PIPELINING\n";
    if (gconfig->smtpOpts.constraints.messageSizeLimit > 0) {
        respText << "SIZE " << std::to_string(gconfig->smtpOpts.constraints.messageSizeLimit) << "\n";
    }
    if (gconfig->smtpOpts.startTls) {
        respText << "STARTTLS\n";
    }
    if (gconfig->auth.use) {
        respText << "AUTH LOGIN PLAIN XOAUTH2\n";
    }
    if (gconfig->dsn.mode != dsn::Options::NEVER) {
        respText << "DSN\n";
    }
    respText << "ENHANCEDSTATUSCODES\n";

    CurrentState = SmtpState::Ehlo;
    WriteRead(ymod_smtpserver::Response(250, respText.str()));
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::Helo& helo) {
    if (!IsTransitionAllowed(CurrentState, SmtpState::Ehlo)) {
        return WriteReadWithUniqId(ymod_smtpserver::Response(503, "Bad sequence of commands.", ymod_smtpserver::EnhancedStatusCode(554)));
    }

    Nwsmtp->SetEhloHost(helo.name);
    std::string resp = boost::asio::ip::host_name();

    CurrentState = SmtpState::Ehlo;
    WriteRead(ymod_smtpserver::Response(250, std::move(resp)));
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::StartTls&) {
    if (connection->isSecure()) {
        return WriteReadWithUniqId(ymod_smtpserver::Response(503, "Bad sequence of commands. StartTls command has been already sent.", ymod_smtpserver::EnhancedStatusCode(554)));
    }

    auto cb = [this, self = shared_from_this()](TErrorCode ec) {
        if (ec) {
            NWLOG_EC_CTX(notice, Nwsmtp->GetContext(), "TLS_HANDSHAKE", fmt::format("error on tls handshake for remote addr {}", RemoteAddr.to_string()), ec);
            return CloseConnection();
        }
        ReadCommand();
    };

    CurrentState = SmtpState::Start;
    WriteResponse(ymod_smtpserver::Response(220, "Go ahead"),
        [this, self = shared_from_this(), cb = std::move(cb)]() {
            connection->tlsHandshake(std::move(cb));
        }
    );
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::Noop&) {
    WriteRead(ymod_smtpserver::Response(250, "Ok", ymod_smtpserver::EnhancedStatusCode(200)));
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::Quit&) {
    WriteResponse(ymod_smtpserver::Response(221, ymod_smtpserver::EnhancedStatusCode(200)),
        [this, self = shared_from_this()]() {
            CloseConnection();
        }
    );
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::Rset&) {
    if (!IsTransitionAllowed(CurrentState, SmtpState::Ehlo)) {
        return WriteReadWithUniqId(ymod_smtpserver::Response(503, "Bad sequence of commands.", ymod_smtpserver::EnhancedStatusCode(554)));
    }

    CurrentState = SmtpState::Ehlo;
    WriteReadWithUniqId(ymod_smtpserver::Response(250, "Flushed", ymod_smtpserver::EnhancedStatusCode(215)));
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::MailFrom& mailfrom) {
    if (!IsTransitionAllowed(CurrentState, SmtpState::MailFrom)) {
        return WriteReadWithUniqId(ymod_smtpserver::Response(503, "Bad sequence of commands.", ymod_smtpserver::EnhancedStatusCode(554)));
    }

    Nwsmtp->Reset();

    NNwSmtp::TMailFromRequest req {
        .MailAddr = mailfrom.addr,
        .Params = mailfrom.params
    };

    auto cb = [this, self = shared_from_this(), &ioContext = IoContext, addr = mailfrom.addr](TErrorCode ec) mutable {
        boost::asio::post(ioContext,
            std::bind(&TSessionImpl::HandleMailFrom, shared_from_this(), std::move(ec), std::move(addr))
        );
    };
    Nwsmtp->MailFrom(std::move(req), std::move(cb));
}

void TSessionImpl::HandleMailFrom(TErrorCode ec, std::string mailFromAddr) {
    if (!ec) {
        CurrentState = SmtpState::MailFrom;
    }
    auto response = NMailFromCommand::ConvertErrorCodeToYmodSmtpAnswer(ec, mailFromAddr);
    WriteReadWithUniqId(std::move(response));
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::RcptTo& rcpt) {
    if (!IsTransitionAllowed(CurrentState, SmtpState::RcptTo)) {
        return WriteReadWithUniqId(ymod_smtpserver::Response(503, "Bad sequence of commands.", ymod_smtpserver::EnhancedStatusCode(554)));
    }

    if (IsRateLimiterEnabled() && !RateLimiter->CheckLimitRcptTo()) {
        auto resp = ymod_smtpserver::Response(451, "Try again later (rate limit)", ymod_smtpserver::EnhancedStatusCode(445));
        std::stringstream responseStream;
        responseStream << resp;
        return WriteClose(std::move(resp));
    }

    if (IsRateLimiterEnabled()) {
        RateLimiter->OnStartRcptTo();
    }

    NWLOG_CTX(notice, Nwsmtp->GetContext(), "RCPT", rcpt.addr);

    TRcptToRequest req {
        .MailAddr = rcpt.addr,
        .Params = rcpt.params
    };
    auto cb = [this, self = shared_from_this(), &ioContext = IoContext, rcptAddr = rcpt.addr](TErrorCode ec, std::string smtpAnswer) {
        boost::asio::post(ioContext,
            std::bind(&TSessionImpl::HandleRcptTo, shared_from_this(), std::move(ec), std::move(smtpAnswer), rcptAddr)
        );
    };
    Nwsmtp->RcptTo(std::move(req), std::move(cb));
}

void TSessionImpl::HandleRcptTo(TErrorCode ec, std::string smtpAnswer, std::string rcptAddr) {
    ymod_smtpserver::Response response;
    if (!ec) {
        response = ymod_smtpserver::Response(250, fmt::format("2.1.5 <{}> recipient ok", rcptAddr));
        CurrentState = SmtpState::RcptTo;
    } else {
        if (ec == make_error_condition(NSmtp::EError::Discarded)) {
            response = ymod_smtpserver::Response(250, fmt::format("2.1.5 <{}> recipient discarded", rcptAddr));
            CurrentState = SmtpState::RcptTo;
        } else {
            response = ConvertStringToYmodSmtpResponse(std::move(smtpAnswer));
        }
    }

    if (IsRateLimiterEnabled()) {
        RateLimiter->OnEndRcptTo();
    }

    WriteReadWithUniqId(std::move(response));
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::Data&) {
    if (!IsTransitionAllowed(CurrentState, SmtpState::Data)) {
        return WriteReadWithUniqId(ymod_smtpserver::Response(503, "Bad sequence of commands.", ymod_smtpserver::EnhancedStatusCode(554)));
    }

    if (IsRateLimiterEnabled() && !RateLimiter->CheckLimitData()) {
        auto resp = ymod_smtpserver::Response(451, "Try again later (rate limit)", ymod_smtpserver::EnhancedStatusCode(445));
        std::stringstream responseStream;
        responseStream << resp;
        return WriteClose(std::move(resp));
    }

    if (IsRateLimiterEnabled()) {
        RateLimiter->OnStartData();
    }

    std::string protocolType = "ESMTP";
    if (connection->isSecure()) {
        protocolType += "S";
    }
    if (Nwsmtp->IsAuthenticated()) {
        protocolType += "A";
    }

    SSL* ssl = connection->isSecure() ? connection->getSsl() : nullptr;

    NNwSmtp::TDataRequest req {
        .Ssl = ssl,
        .ProtocolType = protocolType
    };
    Nwsmtp->Data(std::move(req));

    CurrentState = SmtpState::Data;

    WriteResponse(
        ymod_smtpserver::Response(354, "Start mail input, end with <CRLF>.<CRLF>"),
        [this, self = shared_from_this()]() { ReadMessage(); }
    );
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::Auth& authCmd) {
    if (Nwsmtp->IsAuthenticated()) {
        return WriteReadWithUniqId(
            ymod_smtpserver::Response(503, "Error: already authenticated", ymod_smtpserver::EnhancedStatusCode(551))
        );
    }

    if (gconfig->auth.sslOnly && !Connection->isSecure()) {
        NWLOG_CTX(notice, Nwsmtp->GetContext(), "RECV", fmt::format("denied nossl auth from {} [{}]", RemoteHostName, RemoteAddr.to_string()));

        std::string responseMessage = "Error: Must issue a STARTTLS command first.";
        if (!gconfig->auth.sslMessage.empty()) {
            responseMessage = gconfig->auth.sslMessage;
        }
        return WriteReadWithUniqId(
            ymod_smtpserver::Response(530, std::move(responseMessage), ymod_smtpserver::EnhancedStatusCode(577))
        );
    }

    namespace AuthMethods = ymod_smtpserver::commands::AuthMethods;
    if (authCmd.Method.which() == NAuth::AuthMethod{AuthMethods::NotSupported{}}.which()) {
        return WriteRead(ymod_smtpserver::Response(535, "Error: Method is not supported.", ymod_smtpserver::EnhancedStatusCode(578)));
    }

    Auth->Reset();
    Auth->SetMethod(authCmd.Method);

    if (!authCmd.InitialResponse.empty()) {
        ProcessAuthInput(authCmd.InitialResponse);
    } else {
        std::string welcomeMessage;
        if (boost::get<AuthMethods::Login>(&authCmd.Method)) {
            welcomeMessage = NAuth::ENTER_USERNAME_BASE64;
        } else if (boost::get<AuthMethods::Plain>(&authCmd.Method)) {
            welcomeMessage = NAuth::ENTER_USERNAME_BASE64;
        }
        WriteResponse(
            ymod_smtpserver::Response(334, welcomeMessage),
            [this, self = shared_from_this()]() {
                Connection->readLine(
                    std::bind(&TSessionImpl::HandleAuthLine, self, ph::_1, ph::_2)
                );
            }
        );

    }
}

void TSessionImpl::ProcessAuthInput(const std::string& input) {
    auto res = Auth->Consume(input);
    switch (res.Status) {
        case NAuth::EStatus::BadFormed:
            WriteRead(
                ymod_smtpserver::Response(535, std::move("Error: authentication failed: Invalid format."), ymod_smtpserver::EnhancedStatusCode(578))
            );
            break;
        case NAuth::EStatus::NeedMore:
            WriteResponse(
                ymod_smtpserver::Response(334, res.ReplyToUser),
                [this, self = shared_from_this()]() {
                    Connection->readLine(
                        std::bind(&TSessionImpl::HandleAuthLine, self, ph::_1, ph::_2)
                    );
                }
            );
            break;
        case NAuth::EStatus::Done:
            StartAuth(Auth->GetData());
            break;
    }
}

void TSessionImpl::HandleAuthLine(TErrorCode ec, std::string line) {
    if (ec) {
        return CloseConnection();
    }

    auto trimmed = boost::algorithm::trim_copy(line);
    if (trimmed == "*") {
        WriteRead(ymod_smtpserver::Response(501, std::move("Cancelled.")));
    } else {
        ProcessAuthInput(std::move(trimmed));
    }
}

void TSessionImpl::StartAuth(NAuth::TAuthCredentials data) {
    TAuthData info;
    info.Ip = RemoteAddr.to_string();
    info.Port = std::to_string(Connection->remotePort());
    info.Login = data.Login;
    info.Password = data.Password;
    info.Token = data.Token;
    info.Method = Auth->GetMethod();
    TAuthRequst req {
        .Auth = std::move(info)
    };
    auto cb = [this, self = shared_from_this(), &ioContext = IoContext](TErrorCode ec, NBlackBox::TResponse resp) {
        boost::asio::post(ioContext,
            std::bind(&TSessionImpl::HandleAuth, shared_from_this(), std::move(ec), std::move(resp))
        );
    };
    Nwsmtp->Auth(std::move(req), std::move(cb));
}

void TSessionImpl::HandleAuth(TErrorCode ec, NBlackBox::TResponse resp) {
    auto chkResult = GetAuthResult(ec, resp);
    if (ec) {
        NWLOG_EC_CTX(notice, Nwsmtp->GetContext(), "AUTH", "error on auth: " + chkResult.LogMsg, ec);
    }
    auto smtpResp = ConvertStringToYmodSmtpResponse(chkResult.SmtpResponse);
    WriteReadWithUniqId(std::move(smtpResp));
}

void TSessionImpl::StartDelivery(std::shared_ptr<std::string> msg) {
    auto message = NUtil::MakeSegment(*msg);
    auto size = message.size();
    TDeliveryRequest req {
        .Msg = std::move(message),
        .MsgSize = size
    };

    auto cb = [this, self = shared_from_this(), &ioContext = IoContext](TErrorCode ec, std::string answer) {
        boost::asio::post(ioContext,
            std::bind(&TSessionImpl::HandleDelivery, shared_from_this(), std::move(ec), std::move(answer))
        );
    };
    Nwsmtp->Delivery(std::move(req), std::move(cb));
}

void TSessionImpl::HandleDelivery(TErrorCode ec, std::string answer) {
    using EError = NNwSmtp::NSmtp::EError;

    ymod_smtpserver::Response resp;
    if (!ec) {
        std::string respStr = fmt::format("Ok: queued on {}", boost::asio::ip::host_name());
        resp = ymod_smtpserver::Response(250, respStr, ymod_smtpserver::EnhancedStatusCode(200));
    } else {
        NWLOG_EC_CTX(notice, Nwsmtp->GetContext(), "REPLY", "Error occurred", ec);
        if (ec == make_error_condition(EError::Discarded)) {
            if (!answer.empty()) {
                std::string respStr = fmt::format("{}: queued on {}", answer, boost::asio::ip::host_name());
                resp = ConvertStringToYmodSmtpResponse(std::move(respStr));
            } else {
                std::string respStr = fmt::format("Ok: queued on {}", boost::asio::ip::host_name());
                resp = ymod_smtpserver::Response(250, std::move(respStr), ymod_smtpserver::EnhancedStatusCode(200));
            }
        } else if (ec == make_error_condition(EError::Rejected)) {
            if (!answer.empty()) {
                resp = ConvertStringToYmodSmtpResponse(answer);
            } else {
                std::string respStr = boost::asio::ip::host_name();
                resp = ymod_smtpserver::Response(550, std::move(respStr));
            }
        } else {
            if (!answer.empty()) {
                resp = ConvertStringToYmodSmtpResponse(answer);
            } else {
                std::string respStr = fmt::format("Ok: queued on {}", boost::asio::ip::host_name());
                resp = ymod_smtpserver::Response(451, std::move(respStr), ymod_smtpserver::EnhancedStatusCode(471));
            }
        }
    }
    std::stringstream responseStream;
    responseStream << resp;
    NWLOG_CTX(notice, Nwsmtp->GetContext(), "REPLY", responseStream.str());
    if (IsRateLimiterEnabled()) {
        RateLimiter->OnEndData();
    }
    CurrentState = SmtpState::Ehlo;
    return WriteReadWithUniqId(std::move(resp));
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::Unknown& unknown) {
    NWLOG_CTX(notice, Nwsmtp->GetContext(), "", fmt::format("unknown error '{}'", unknown.ctx));
    if (++UnrecognizedCommandsCounter >= gconfig->smtpOpts.unrecognizedCommandsMaxCount) {
        auto resp = ymod_smtpserver::Response(502, "Too many unrecognized commands, goodbye", ymod_smtpserver::EnhancedStatusCode(551));
        std::stringstream responseStream;
        responseStream << resp;
        NWLOG_CTX(notice, Nwsmtp->GetContext(), "REPLY", responseStream.str());
        WriteClose(resp);
    } else {
        WriteRead(ymod_smtpserver::Response(502, ymod_smtpserver::EnhancedStatusCode(551)));
    }
}

void TSessionImpl::operator()(const ymod_smtpserver::commands::SyntaxError& syntaxError) {
    NWLOG_CTX(notice, Nwsmtp->GetContext(), "", fmt::format("syntax error '{}'", syntaxError.ctx));
    WriteRead(ymod_smtpserver::Response(555, ymod_smtpserver::EnhancedStatusCode(552)));
}

std::string TSessionImpl::GetUniqId() const {
    return Nwsmtp->GetContext()->GetSessionId();
}

void TSessionImpl::ReadCommand() {
    auto cb = [this, self = shared_from_this()](TErrorCode ec, ymod_smtpserver::Command cmd) mutable {
        if (ec) {
            return CloseConnection();
        }
        return boost::apply_visitor(*this, cmd);
    };
    connection->readCommand(std::move(cb));
}

void TSessionImpl::ReadMessage() {
    auto cb = [this, self = shared_from_this()](TErrorCode ec, std::shared_ptr<std::string> msg) {
        if (ec) {
            NWLOG_EC_CTX(error, Nwsmtp->GetContext(), "RECV", "failed to read message", std::move(ec));
            return CloseConnection();
        }
        StartDelivery(msg);
    };
    connection->readMessage(std::move(cb));
}

void TSessionImpl::CloseConnection() {
    if (IsSessionTimeLimitEnabled()) {
        SessionTimer.cancel();
    }
    NWLOG_CTX(notice, Nwsmtp->GetContext(), "RECV", fmt::format("disconnect from {} [{}]", RemoteHostName, RemoteAddr.to_string()));
    Connection->close();
}

bool TSessionImpl::IsSessionTimeExpired() const {
    return IsSessionTimeLimitEnabled() && SessionTimeExpired.load();
}

bool TSessionImpl::IsSessionTimeLimitEnabled() const {
    return gconfig->smtpOpts.sessionTimeLimit.use
        && gconfig->smtpOpts.sessionTimeLimit.timeout.count() > 0;
}

bool TSessionImpl::IsRateLimiterEnabled() const {
    return gconfig->smtpOpts.useRateLimiter;
};

void TSessionImpl::WriteRead(ymod_smtpserver::Response response) {
    WriteResponse(std::move(response), [this, self = shared_from_this()]() { ReadCommand(); });
}

void TSessionImpl::WriteClose(ymod_smtpserver::Response response) {
    WriteResponse(std::move(response), [this, self = shared_from_this()]() { CloseConnection(); });
}

void TSessionImpl::WriteReadWithUniqId(ymod_smtpserver::Response response) {
    WriteResponseWithUniqId(std::move(response), [this, self = shared_from_this()]() { ReadCommand(); });
}

SessionPtr TSessionFactoryImpl::create(ConnectionPtr connection) {
    return std::make_shared<TSessionImpl>(std::move(connection), *Reactor->io());
}

}
