#pragma once

#include "smtp_fsm.h"

#include <mail/nwsmtp/src/types.h>
#include <mail/nwsmtp/src/auth.h>
#include <mail/nwsmtp/src/utils.h>

#include <mail/nwsmtp/src/rbl/http_client.h>
#include <mail/nwsmtp/src/resolver/client.h>
#include <mail/nwsmtp/src/resolver/client_impl.h>
#include <mail/nwsmtp/src/resolver/resolver.h>
#include <mail/nwsmtp/src/nwsmtp/impl.h>
#include <mail/nwsmtp/src/auth/interface.h>
#include <mail/nwsmtp/src/rate_limiter/impl.h>

#include <ymod_smtpserver/connection.h>
#include <ymod_smtpserver/session.h>
#include <ymod_smtpserver/server.h>

#include <yplatform/reactor.h>

#include <boost/asio/io_context.hpp>

#include <memory>

namespace NNwSmtp::NSmtpServer {

using ymod_smtpserver::ConnectionPtr;
using ymod_smtpserver::SessionPtr;

class TSessionImpl
    : public ymod_smtpserver::Session
    , private boost::static_visitor<>
    , public std::enable_shared_from_this<TSessionImpl>
{
public:
    TSessionImpl(ConnectionPtr connection, boost::asio::io_context& ioContext);

    void start();

    template <typename TCommand>
    void operator()(TCommand) {
        WriteRead(ymod_smtpserver::Response(502, ymod_smtpserver::EnhancedStatusCode(551)));
    }

    void operator()(const ymod_smtpserver::commands::Ehlo& ehlo);

    void operator()(const ymod_smtpserver::commands::Helo& helo);

    void operator()(const ymod_smtpserver::commands::StartTls&);

    void operator()(const ymod_smtpserver::commands::Noop&);

    void operator()(const ymod_smtpserver::commands::Quit&);

    void operator()(const ymod_smtpserver::commands::Rset&);

    void operator()(const ymod_smtpserver::commands::MailFrom& mailfrom);

    void operator()(const ymod_smtpserver::commands::RcptTo& rcpt);

    void operator()(const ymod_smtpserver::commands::Data&);

    void operator()(const ymod_smtpserver::commands::Auth&);

    void operator()(const ymod_smtpserver::commands::Unknown& /*unknown*/);

    void operator()(const ymod_smtpserver::commands::SyntaxError& /*syntaxError*/);

private:
    void HandleIpResolve(TErrorCode ec, std::string hostName);

    void HandleSoRbl(TErrorCode ec, bool foundInSpam);

    void CheckRateLimit();

    void HandleCheckRateLimit(TErrorCode ec);

    void HandleMailFrom(TErrorCode ec, std::string mailFromAddr);

    void HandleRcptTo(TErrorCode ec, std::string smtpAnswer, std::string rcptAddr);

    void StartDelivery(std::shared_ptr<std::string> msg);

    void HandleDelivery(TErrorCode ec, std::string answer);

    void ProcessAuthInput(const std::string& input);
    void HandleAuthLine(TErrorCode ec, std::string line);

    void StartAuth(NAuth::TAuthCredentials data);
    void HandleAuth(TErrorCode ec, NBlackBox::TResponse resp);

    std::string GetUniqId() const;

private:
    void ReadCommand();

    void ReadMessage();

    void WriteRead(ymod_smtpserver::Response response);

    void WriteClose(ymod_smtpserver::Response response);

    void WriteReadWithUniqId(ymod_smtpserver::Response response);

    void CloseConnection();

    template <typename Handler>
    void WriteResponse(ymod_smtpserver::Response response, Handler handler) {
        if (IsSessionTimeExpired()) {
            return CloseConnection();
        }

        auto cb = [this, self = shared_from_this(), handler = std::move(handler)](TErrorCode ec, std::size_t) {
            if (ec) {
                return CloseConnection();
            }
            handler();
        };
        connection->writeResponse(std::move(response), std::move(cb));
    }

    template <typename Handler>
    void WriteResponseWithUniqId(ymod_smtpserver::Response response, Handler handler) {
        response.text += " " + GetUnixTimestamp() + "-" + GetUniqId();
        return WriteResponse(std::move(response), std::move(handler));
    }

    bool IsSessionTimeExpired() const;
    bool IsSessionTimeLimitEnabled() const;

    bool IsRateLimiterEnabled() const;

private:
    ConnectionPtr Connection;
    boost::asio::io_context& IoContext;

    NSO::NRBL::THttpClientPtr SoRblChecker;
    NDns::TResolverClientPtr ResolverClient;
    NNwSmtp::TNwSmtpPtr Nwsmtp;

    SmtpState CurrentState = SmtpState::Start;
    boost::asio::ip::address RemoteAddr;
    std::string RemoteHostName;

    NAuth::TAuthCredentialsExtractorPtr Auth;

    yplatform::time_traits::timer SessionTimer;
    std::atomic<bool> SessionTimeExpired = false;
    std::size_t UnrecognizedCommandsCounter = 0;

    NRateLimiter::TRequestsRateLimiterPtr RateLimiter;
};

class TSessionFactoryImpl : public ymod_smtpserver::SessionFactory {
public:
    TSessionFactoryImpl(yplatform::reactor_ptr reactor)
        : Reactor(reactor)
    {}

    SessionPtr create(ConnectionPtr connection);

private:
    yplatform::reactor_ptr Reactor;
};

} // namespace NNwSmtp::NSmtpServer
