#include "config.h"

#include <boost/algorithm/string/case_conv.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/tokenizer.hpp>

#include <fstream>

namespace NNwSmtp::NSO {

namespace {

void LoadRejectedTypes(const std::string& path, std::vector<std::set<std::string>>& types) {
    std::ifstream is(path.c_str());
    if (!is.good()) {
        throw std::runtime_error("failed to read " + path);
    }
    std::string line;
    while (std::getline(is, line)) {
        boost::trim(line);
        if (line.empty() || line[0] == '#') {
            continue;
        }
        boost::char_separator<char> sep(", \t");
        boost::tokenizer<boost::char_separator<char>> toks(line, sep);
        std::set<std::string> group;
        for (const auto& tok : toks) {
            group.insert(boost::to_lower_copy(tok));
        }
        if (group.empty()) {
            continue;
        }
        types.push_back(std::move(group));
    }
}

ESpamPolicy SpamPolicyFromString(const std::string& spamPolicy) {
    if (spamPolicy == "discard") {
        return ESpamPolicy::DISCARD;
    } else if (spamPolicy == "reject") {
        return ESpamPolicy::REJECT;
    } else if (spamPolicy == "field_mark") {
        return ESpamPolicy::FIELD_MARK;
    } else {
        throw std::runtime_error("no such so action: " + spamPolicy);
    }
}

}

void TOptions::Read(const yplatform::ptree& pt) {
    auto options = pt.equal_range("internal_headers");
    for (auto header = options.first; header != options.second; ++header) {
        InternalHeaders.emplace(boost::to_lower_copy(header->second.data()));
    }

    if (auto option = pt.get_optional<std::string>("reply_text_malicious")) {
        ReplyTextMalicious = option.value();
    }
    if (auto option = pt.get_optional<std::string>("reply_text_spam")) {
        ReplyTextSpam = option.value();
    }
    if (auto option = pt.get_optional<std::string>("reply_text_discard")) {
        ReplyTextDiscard = option.value();
    }
    ClusterName = pt.get<std::string>("cluster_name");
    SoClient = pt.get<std::string>("client", "so_in_client");

    Check = pt.get<bool>("check", false);
    DryRun = pt.get<bool>("dry_run", false);
    AddXspamFlag = pt.get<bool>("add_xspam_flag", false);

    CheckInactivity = pt.get<bool>("inactivity.check", true);
    RejectForInactive = pt.get<bool>("inactivity.reject", false);
    InactivityThreshold = pt.get<std::uint32_t>("inactivity.threshold", 100);
    NewUserPeriod = std::chrono::days(pt.get<std::uint32_t>("inactivity.new_user_period", 0));
    if (auto option = pt.get_optional<std::string>("inactivity.rejected_types")) {
        LoadRejectedTypes(option.value(), RejectedTypes);
    }

    ActionSpam = SpamPolicyFromString(pt.get<std::string>("action.spam", "field_mark"));
    ActionMalicious = SpamPolicyFromString(pt.get<std::string>("action.malicious", "reject"));
}

} // namespace NNwSmtp::NSO
